/**
 * Toolbar Module
 * จัดการแถบเครื่องมือและการทำงานของปุ่ม
 */
(function(global) {
  'use strict';

  /**
   * คลาส Toolbar
   * จัดการแถบเครื่องมือและการทำงานของปุ่ม
   */
  class Toolbar {
    constructor(editor) {
      this.editor = editor;
      this.toolbar = null;
      this.buttons = {};
      this.shortcuts = {};
    }

    /**
     * เริ่มต้นโมดูลแถบเครื่องมือ
     */
    init() {
      this.setupEventListeners();
      this.setupKeyboardShortcuts();

      if (this.editor.config.debug) {
        console.log('Toolbar เริ่มต้นแล้ว');
      }
    }

    /**
     * ตั้งค่า event listeners
     */
    setupEventListeners() {
      // ฟังเหตุการณ์การเปลี่ยนแปลงโหมดตัวแก้ไข
      this.editor.on('editor:mode-changed', (data) => {
        this.updateToolbarState(data.mode);
      });

      // ฟังเหตุการณ์การเริ่มต้นแก้ไขเนื้อหา
      this.editor.on('content:editing-started', () => {
        this.showContentToolbar();
      });

      // ฟังเหตุการณ์การสิ้นสุดการแก้ไขเนื้อหา
      this.editor.on('content:editing-ended', () => {
        this.hideContentToolbar();
      });
    }

    /**
     * ตั้งค่าคีย์บอร์ดลัด
     */
    setupKeyboardShortcuts() {
      // กำหนดคีย์บอร์ดลัด
      this.shortcuts = {
        'Ctrl+S': () => {
          this.editor.saveTemplate();
        },
        'Ctrl+Z': () => {
          this.editor.emit('toolbar:undo');
        },
        'Ctrl+Y': () => {
          this.editor.emit('toolbar:redo');
        },
        'Ctrl+E': () => {
          this.editor.emit('toolbar:edit');
        },
        'Ctrl+P': () => {
          this.editor.emit('toolbar:preview');
        },
        'Escape': () => {
          if (this.editor.state === 'edit') {
            this.editor.emit('toolbar:preview');
          }
        }
      };

      // เพิ่ม event listener สำหรับคีย์บอร์ดลัด
      document.addEventListener('keydown', (e) => {
        const key = this.getShortcutKey(e);
        if (this.shortcuts[key]) {
          e.preventDefault();
          this.shortcuts[key]();
        }
      });
    }

    /**
     * รับคีย์ลัด
     */
    getShortcutKey(e) {
      const parts = [];

      if (e.ctrlKey || e.metaKey) {
        parts.push('Ctrl');
      }

      if (e.shiftKey) {
        parts.push('Shift');
      }

      if (e.altKey) {
        parts.push('Alt');
      }

      parts.push(e.key);

      return parts.join('+');
    }

    /**
     * อัปเดตสถานะแถบเครื่องมือ
     */
    updateToolbarState(mode) {
      const editBtn = document.getElementById('toolbar-edit-btn');
      const previewBtn = document.getElementById('toolbar-preview-btn');

      if (mode === 'edit') {
        if (editBtn) editBtn.classList.add('active');
        if (previewBtn) previewBtn.classList.remove('active');
      } else if (mode === 'preview') {
        if (editBtn) editBtn.classList.remove('active');
        if (previewBtn) previewBtn.classList.add('active');
      }
    }

    /**
     * แสดงแถบเครื่องมือเนื้อหา
     */
    showContentToolbar() {
      const contentToolbar = document.getElementById('editor-text-toolbar');
      if (contentToolbar) {
        contentToolbar.style.display = 'flex';
        setTimeout(() => {
          contentToolbar.classList.add('editor-toolbar-active');
        }, 10);
      }
    }

    /**
     * ซ่อนแถบเครื่องมือเนื้อหา
     */
    hideContentToolbar() {
      const contentToolbar = document.getElementById('editor-text-toolbar');
      if (contentToolbar) {
        contentToolbar.classList.remove('editor-toolbar-active');
        setTimeout(() => {
          contentToolbar.style.display = 'none';
        }, 200);
      }
    }
  }

  // เปิดเผยคลาส Toolbar ทั่วโลก
  global.Toolbar = Toolbar;

})(typeof window !== 'undefined' ? window : this);