<?php
/**
 * api.php - ไฟล์ API สำหรับจัดการโครงสร้างต้นไม้
 */

// ตั้งค่าการตอบกลับเป็น JSON
header('Content-Type: application/json');

// นำเข้าไฟล์ TreeAPI
require_once 'TreeAPI.php';

// สร้างอินสแตนซ์ของคลาส TreeAPI
$treeAPI = new TreeAPI();

// สร้างฟังก์ชันตอบกลับ
function sendResponse($status, $data = [], $message = '')
{
    $response = [
        'status' => $status,
        'data' => $data
    ];

    if (!empty($message)) {
        $response['message'] = $message;
    }

    echo json_encode($response);
    exit;
}

// ตรวจสอบว่า method เป็น POST หรือไม่
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // รับข้อมูล JSON จาก request body
    $jsonData = file_get_contents('php://input');
    $data = json_decode($jsonData, true);

    // ถ้า JSON ไม่ถูกต้อง
    if ($jsonData && !$data) {
        sendResponse('error', [], 'ข้อมูล JSON ไม่ถูกต้อง');
    }
}

// อ่านและตรวจสอบ action
$action = isset($_GET['action']) ? $_GET['action'] : '';

switch ($action) {
    case 'get_tree':
        // ดึงข้อมูลต้นไม้ทั้งหมด
        $treeData = $treeAPI->getTreeStructure();
        sendResponse('success', $treeData);
        break;

    case 'export_tree':
        // ดึงข้อมูลต้นไม้ในรูปแบบของ TreeManager
        $treeData = $treeAPI->exportToTreeManager();
        sendResponse('success', $treeData);
        break;

    case 'import_tree':
        // นำเข้าข้อมูลต้นไม้จาก TreeManager
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            sendResponse('error', [], 'Method ไม่ถูกต้อง สำหรับการนำเข้าต้องใช้ POST');
        }

        if (empty($data)) {
            sendResponse('error', [], 'ไม่พบข้อมูลที่ต้องการนำเข้า');
        }

        $result = $treeAPI->importFromTreeManager($data);

        if ($result) {
            sendResponse('success', [], 'นำเข้าข้อมูลสำเร็จ');
        } else {
            sendResponse('error', [], 'เกิดข้อผิดพลาดในการนำเข้าข้อมูล');
        }
        break;

    case 'get_node':
        // ดึงข้อมูลโหนดตาม ID
        $id = isset($_GET['id']) ? intval($_GET['id']) : 0;

        if ($id <= 0) {
            sendResponse('error', [], 'ID ไม่ถูกต้อง');
        }

        $node = $treeAPI->getNode($id);

        if ($node) {
            sendResponse('success', $node);
        } else {
            sendResponse('error', [], 'ไม่พบโหนดที่ต้องการ');
        }
        break;

    case 'get_node_by_external_id':
        // ดึงข้อมูลโหนดตาม external_id
        $externalId = isset($_GET['external_id']) ? $_GET['external_id'] : '';

        if (empty($externalId)) {
            sendResponse('error', [], 'External ID ไม่ถูกต้อง');
        }

        $node = $treeAPI->getNodeByExternalId($externalId);

        if ($node) {
            sendResponse('success', $node);
        } else {
            sendResponse('error', [], 'ไม่พบโหนดที่ต้องการ');
        }
        break;

    case 'add_node':
        // เพิ่มโหนดใหม่
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            sendResponse('error', [], 'Method ไม่ถูกต้อง สำหรับการเพิ่มต้องใช้ POST');
        }

        if (empty($data['name']) || !isset($data['level'])) {
            sendResponse('error', [], 'ข้อมูลไม่ครบถ้วน ต้องระบุชื่อและระดับ');
        }

        $name = $data['name'];
        $level = intval($data['level']);
        $parentId = isset($data['parent_id']) ? intval($data['parent_id']) : null;
        $externalId = isset($data['external_id']) ? $data['external_id'] : null;
        $additionalData = isset($data['data']) ? json_encode($data['data']) : null;

        $result = $treeAPI->addNode($name, $level, $parentId, $externalId, $additionalData);

        if ($result) {
            $newNode = $treeAPI->getNode($result);
            sendResponse('success', $newNode, 'เพิ่มโหนดสำเร็จ');
        } else {
            sendResponse('error', [], 'เกิดข้อผิดพลาดในการเพิ่มโหนด');
        }
        break;

    case 'update_node':
        // อัพเดตโหนด
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            sendResponse('error', [], 'Method ไม่ถูกต้อง สำหรับการอัพเดตต้องใช้ POST');
        }

        if (empty($data['id'])) {
            sendResponse('error', [], 'ต้องระบุ ID ของโหนดที่ต้องการอัพเดต');
        }

        $id = intval($data['id']);

        // ตรวจสอบว่ามีโหนดที่ต้องการอัพเดตหรือไม่
        $node = $treeAPI->getNode($id);

        if (!$node) {
            sendResponse('error', [], 'ไม่พบโหนดที่ต้องการอัพเดต');
        }

        // สร้างข้อมูลที่ต้องการอัพเดต
        $updateData = [];

        if (isset($data['name'])) {
            $updateData['name'] = $data['name'];
        }

        if (isset($data['level'])) {
            $updateData['level'] = intval($data['level']);
        }

        if (isset($data['parent_id'])) {
            $updateData['parent_id'] = $data['parent_id'] ? intval($data['parent_id']) : null;
        }

        if (isset($data['external_id'])) {
            $updateData['external_id'] = $data['external_id'];
        }

        if (isset($data['data'])) {
            $updateData['data'] = json_encode($data['data']);
        }

        if (empty($updateData)) {
            sendResponse('error', [], 'ไม่มีข้อมูลที่ต้องการอัพเดต');
        }

        $result = $treeAPI->updateNode($id, $updateData);

        if ($result) {
            $updatedNode = $treeAPI->getNode($id);
            sendResponse('success', $updatedNode, 'อัพเดตโหนดสำเร็จ');
        } else {
            sendResponse('error', [], 'เกิดข้อผิดพลาดในการอัพเดตโหนด');
        }
        break;

    case 'delete_node':
        // ลบโหนด
        $id = isset($_GET['id']) ? intval($_GET['id']) : 0;

        if ($id <= 0) {
            sendResponse('error', [], 'ID ไม่ถูกต้อง');
        }

        // ตรวจสอบว่ามีโหนดที่ต้องการลบหรือไม่
        $node = $treeAPI->getNode($id);

        if (!$node) {
            sendResponse('error', [], 'ไม่พบโหนดที่ต้องการลบ');
        }

        $result = $treeAPI->deleteNode($id);

        if ($result) {
            sendResponse('success', [], 'ลบโหนดและลูกหลานสำเร็จ');
        } else {
            sendResponse('error', [], 'เกิดข้อผิดพลาดในการลบโหนด');
        }
        break;

    case 'move_node':
        // ย้ายโหนด
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            sendResponse('error', [], 'Method ไม่ถูกต้อง สำหรับการย้ายโหนดต้องใช้ POST');
        }

        if (empty($data['id']) || !isset($data['direction'])) {
            sendResponse('error', [], 'ข้อมูลไม่ครบถ้วน ต้องระบุ ID และทิศทาง');
        }

        $id = intval($data['id']);
        $direction = $data['direction'];

        // ตรวจสอบทิศทาง
        if ($direction !== 'up' && $direction !== 'down') {
            sendResponse('error', [], 'ทิศทางไม่ถูกต้อง ต้องเป็น up หรือ down');
        }

        $result = $treeAPI->moveNode($id, $direction);

        if ($result) {
            sendResponse('success', [], 'ย้ายโหนดสำเร็จ');
        } else {
            sendResponse('error', [], 'เกิดข้อผิดพลาดในการย้ายโหนด');
        }
        break;

    case 'move_node_to_parent':
        // ย้ายโหนดไปอยู่ภายใต้พ่อใหม่
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            sendResponse('error', [], 'Method ไม่ถูกต้อง สำหรับการย้ายโหนดต้องใช้ POST');
        }

        if (empty($data['id']) || !isset($data['new_level'])) {
            sendResponse('error', [], 'ข้อมูลไม่ครบถ้วน ต้องระบุ ID และระดับใหม่');
        }

        $id = intval($data['id']);
        $newParentId = isset($data['new_parent_id']) ? intval($data['new_parent_id']) : null;
        $newLevel = intval($data['new_level']);

        $result = $treeAPI->moveNodeToParent($id, $newParentId, $newLevel);

        if ($result) {
            sendResponse('success', [], 'ย้ายโหนดสำเร็จ');
        } else {
            sendResponse('error', [], 'เกิดข้อผิดพลาดในการย้ายโหนด');
        }
        break;

    case 'search_nodes':
        // ค้นหาโหนด
        $searchTerm = isset($_GET['term']) ? $_GET['term'] : '';

        if (empty($searchTerm)) {
            sendResponse('error', [], 'ต้องระบุคำค้นหา');
        }

        $results = $treeAPI->searchNodes($searchTerm);
        sendResponse('success', $results);
        break;

    default:
        // Action ไม่ถูกต้อง
        sendResponse('error', [], 'Action ไม่ถูกต้องหรือไม่ได้ระบุ');
        break;
}
