// Order Management logic
class OrderManagement {
  constructor() {
    this.dbHelper = dbHelper;
    this.orders = [];
    this.currentFilter = 'all';
    this.currentOrderId = null;

    // Initialize modals
    this.orderDetailsModal = new Modal('orderDetailsModal');
    this.updateStatusModal = new Modal('updateStatusModal');

    // Initialize event listeners
    this.initEventListeners();

    // Start auto-refresh
    this.startAutoRefresh();
  }

  async init() {
    try {
      // Initialize database
      await this.dbHelper.init();

      // Initialize with mock data if needed
      await this.dbHelper.initializeWithMockData();

      // Load orders
      await this.loadOrders();

      console.log('Order Management initialized successfully');
    } catch (error) {
      console.error('Error initializing Order Management:', error);
    }
  }

  initEventListeners() {
    // Filter tabs
    const filterTabs = document.querySelectorAll('.filter-tab');
    filterTabs.forEach(tab => {
      tab.addEventListener('click', () => {
        // Remove active class from all tabs
        filterTabs.forEach(t => t.classList.remove('active'));

        // Add active class to clicked tab
        tab.classList.add('active');

        // Filter orders
        this.currentFilter = tab.getAttribute('data-filter');
        this.filterOrders();
      });
    });

    // Refresh button
    const refreshBtn = document.getElementById('refreshOrdersBtn');
    if (refreshBtn) {
      refreshBtn.addEventListener('click', () => {
        this.loadOrders();
        toast.show('รีเฟรชข้อมูลสำเร็จ');
      });
    }

    // Close order details button
    const closeOrderDetailsBtn = document.getElementById('closeOrderDetailsBtn');
    if (closeOrderDetailsBtn) {
      closeOrderDetailsBtn.addEventListener('click', () => {
        this.orderDetailsModal.close();
      });
    }

    // Update order status button
    const updateOrderStatusBtn = document.getElementById('updateOrderStatusBtn');
    if (updateOrderStatusBtn) {
      updateOrderStatusBtn.addEventListener('click', () => {
        this.orderDetailsModal.close();
        this.updateStatusModal.open();
      });
    }

    // Status options
    const statusOptions = document.querySelectorAll('.status-option');
    statusOptions.forEach(option => {
      option.addEventListener('click', () => {
        // Remove selected class from all options
        statusOptions.forEach(o => o.classList.remove('selected'));

        // Add selected class to clicked option
        option.classList.add('selected');
      });
    });

    // Cancel update button
    const cancelUpdateBtn = document.getElementById('cancelUpdateBtn');
    if (cancelUpdateBtn) {
      cancelUpdateBtn.addEventListener('click', () => {
        this.updateStatusModal.close();
      });
    }

    // Confirm update button
    const confirmUpdateBtn = document.getElementById('confirmUpdateBtn');
    if (confirmUpdateBtn) {
      confirmUpdateBtn.addEventListener('click', () => {
        this.updateOrderStatus();
      });
    }
  }

  async loadOrders() {
    try {
      // Get orders from IndexedDB
      const ordersData = await this.dbHelper.getAll('orders');
      this.orders = ordersData.map(order => new Order(order));

      // Sort orders by date (newest first)
      this.orders.sort((a, b) => new Date(b.date) - new Date(a.date));

      // Render orders
      this.renderOrders();
    } catch (error) {
      console.error('Error loading orders:', error);
    }
  }

  renderOrders() {
    const ordersList = document.getElementById('ordersList');
    if (ordersList) {
      if (this.orders.length === 0) {
        ordersList.innerHTML = '<p>ไม่มีคำสั่งซื้อ</p>';
      } else {
        ordersList.innerHTML = this.orders.map(order => order.renderCard()).join('');

        // Add event listeners to order cards
        this.attachOrderEventListeners();
      }
    }
  }

  filterOrders() {
    let filteredOrders = this.orders;

    // Filter by status
    if (this.currentFilter !== 'all') {
      filteredOrders = filteredOrders.filter(order => order.status === this.currentFilter);
    }

    // Render filtered orders
    const ordersList = document.getElementById('ordersList');
    if (ordersList) {
      if (filteredOrders.length === 0) {
        ordersList.innerHTML = '<p>ไม่มีคำสั่งซื้อที่ตรงกับเงื่อนไข</p>';
      } else {
        ordersList.innerHTML = filteredOrders.map(order => order.renderCard()).join('');

        // Add event listeners to order cards
        this.attachOrderEventListeners();
      }
    }
  }

  attachOrderEventListeners() {
    // View order buttons
    document.querySelectorAll('.view-order-btn').forEach(button => {
      button.addEventListener('click', (e) => {
        const orderId = e.target.getAttribute('data-id');
        this.viewOrderDetails(orderId);
      });
    });

    // Update status buttons
    document.querySelectorAll('.update-status-btn').forEach(button => {
      button.addEventListener('click', (e) => {
        const orderId = e.target.getAttribute('data-id');
        this.openUpdateStatusModal(orderId);
      });
    });
  }

  viewOrderDetails(orderId) {
    // Find order
    const order = this.orders.find(o => o.id === orderId);
    if (!order) return;

    // Update order details modal
    const orderDetailsContent = document.getElementById('orderDetailsContent');
    orderDetailsContent.innerHTML = order.renderDetails();

    // Store current order ID
    this.currentOrderId = orderId;

    // Open modal
    this.orderDetailsModal.open();
  }

  openUpdateStatusModal(orderId) {
    // Find order
    const order = this.orders.find(o => o.id === orderId);
    if (!order) return;

    // Store current order ID
    this.currentOrderId = orderId;

    // Select current status
    document.querySelectorAll('.status-option').forEach(option => {
      option.classList.remove('selected');
      if (option.getAttribute('data-status') === order.status) {
        option.classList.add('selected');
      }
    });

    // Open modal
    this.updateStatusModal.open();
  }

  async updateOrderStatus() {
    try {
      if (!this.currentOrderId) return;

      // Get selected status
      const selectedStatus = document.querySelector('.status-option.selected');
      if (!selectedStatus) {
        toast.show('กรุณาเลือกสถานะ', 'warning');
        return;
      }

      const newStatus = selectedStatus.getAttribute('data-status');

      // Find order
      const order = this.orders.find(o => o.id === this.currentOrderId);
      if (!order) return;

      // Update order status
      order.status = newStatus;

      // Update in IndexedDB
      await this.dbHelper.update('orders', order);

      // Close modal
      this.updateStatusModal.close();

      // Reload orders
      await this.loadOrders();

      // Show success message
      toast.show('อัปเดตสถานะสำเร็จ');

      console.log('Order status updated successfully:', order);
    } catch (error) {
      console.error('Error updating order status:', error);
      toast.show('เกิดข้อผิดพลาดในการอัปเดตสถานะ', 'error');
    }
  }

  startAutoRefresh() {
    // Refresh orders every 5 seconds
    setInterval(() => {
      this.loadOrders();
    }, 5000);
  }
}

// Initialize Order Management when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
  const orderManagement = new OrderManagement();
  orderManagement.init();
});