/* Service Worker Registration */
if ('serviceWorker' in navigator) {
  window.addEventListener('load', function() {
    navigator.serviceWorker.register('/sw.js');
  });
}

/* Enhanced Canvas Drawing Functions */
function drawTextWithShadow(text, x, y, shadowColor = 'rgba(0,0,0,0.5)') {
  const fontSize = Math.max(canvas.width / 15, 24);

  // Draw shadow
  ctx.fillStyle = shadowColor;
  ctx.fillText(text + '  ', x + 2, y + 2);

  // Draw outline (stroke)
  ctx.strokeStyle = '#000000';
  ctx.lineWidth = fontSize / 8;
  ctx.strokeText(text, x, y);

  // Draw text fill
  ctx.fillStyle = '#FFFFFF';
  ctx.fillText(text, x, y);
}

/* Advanced Text Wrapping */
function drawWrappedText(text, x, y, maxWidth, lineHeight) {
  const words = text.split(' ');
  let line = '';
  let testLine = '';
  let metrics = null;
  let testWidth = 0;

  for (let n = 0; n < words.length; n++) {
    testLine = line + words[n] + ' ';
    metrics = ctx.measureText(testLine);
    testWidth = metrics.width;

    if (testWidth > maxWidth && n > 0) {
      drawTextWithOutline(line, x, y);
      line = words[n] + ' ';
      y += lineHeight;
    } else {
      line = testLine;
    }
  }
  drawTextWithOutline(line, x, y);
}

/* Image Filter Effects */
function applyImageFilter(filter) {
  if (!currentImage) return;

  ctx.filter = filter;
  ctx.clearRect(0, 0, canvas.width, canvas.height);
  ctx.drawImage(currentImage, 0, 0, canvas.width, canvas.height);
  ctx.filter = 'none';

  // Redraw text
  const topText = document.getElementById('top-text').value.toUpperCase();
  const bottomText = document.getElementById('bottom-text').value.toUpperCase();

  const fontSize = Math.max(canvas.width / 15, 24);
  ctx.font = `bold ${fontSize}px Impact, Arial, sans-serif`;
  ctx.textAlign = 'center';
  ctx.textBaseline = 'middle';

  if (topText) {
    drawTextWithOutline(topText, canvas.width / 2, fontSize / 2 + 20);
  }

  if (bottomText) {
    drawTextWithOutline(bottomText, canvas.width / 2, canvas.height - fontSize / 2 - 20);
  }
}

/* Drag and Drop Support */
function setupDragAndDrop() {
  const dropZone = document.getElementById('meme-canvas');

  dropZone.addEventListener('dragover', function(e) {
    e.preventDefault();
    e.stopPropagation();
    dropZone.style.border = '3px dashed var(--accent-color)';
  });

  dropZone.addEventListener('dragleave', function(e) {
    e.preventDefault();
    e.stopPropagation();
    dropZone.style.border = '2px solid var(--border-color)';
  });

  dropZone.addEventListener('drop', function(e) {
    e.preventDefault();
    e.stopPropagation();
    dropZone.style.border = '2px solid var(--border-color)';

    const files = e.dataTransfer.files;
    if (files.length > 0) {
      const file = files[0];
      if (validateFile(file)) {
        handleFileLoad(file);
      }
    }
  });
}

function handleFileLoad(file) {
  const reader = new FileReader();
  reader.onload = function(e) {
    const img = new Image();
    img.onload = function() {
      currentImage = img;

      const maxWidth = 600;
      const maxHeight = 600;
      let {width, height} = img;

      if (width > maxWidth || height > maxHeight) {
        const ratio = Math.min(maxWidth / width, maxHeight / height);
        width *= ratio;
        height *= ratio;
      }

      canvas.width = width;
      canvas.height = height;
      updateMeme();
    };
    img.src = e.target.result;
  };
  reader.readAsDataURL(file);
}

/* Performance Optimization */
let updateTimeout;
function debouncedUpdateMeme() {
  clearTimeout(updateTimeout);
  updateTimeout = setTimeout(updateMeme, 100);
}

/* Copy to Clipboard */
async function copyMemeToClipboard() {
  if (!currentImage) {
    alert('กรุณาเลือกรูปภาพหรือเทมเพลตก่อน');
    return;
  }

  try {
    canvas.toBlob(async (blob) => {
      const item = new ClipboardItem({'image/png': blob});
      await navigator.clipboard.write([item]);

      // Show success message
      showNotification('คัดลอกมีมไปยังคลิปบอร์ดแล้ว! 📋');
    });
  } catch (err) {
    console.error('Failed to copy: ', err);
    alert('ไม่สามารถคัดลอกได้ กรุณาใช้การดาวน์โหลดแทน');
  }
}

/* Show Notification */
function showNotification(message, type = 'success') {
  const notification = document.createElement('div');
  notification.className = `notification ${type}`;
  notification.textContent = message;
  notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: var(--accent-color);
        color: var(--primary-bg);
        padding: 1rem 1.5rem;
        border-radius: 5px;
        z-index: 9999;
        font-weight: bold;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
        animation: slideIn 0.3s ease-out;
    `;

  document.body.appendChild(notification);

  setTimeout(() => {
    notification.style.animation = 'slideOut 0.3s ease-in';
    setTimeout(() => {
      document.body.removeChild(notification);
    }, 300);
  }, 3000);
}

/* Analytics Events (Optional) */
function trackEvent(eventName, properties = {}) {
  // You can integrate with Google Analytics or other analytics services
}

/* Initialize additional features */
document.addEventListener('DOMContentLoaded', function() {
  setupDragAndDrop();

  // Update text inputs to use debounced update
  document.getElementById('top-text').addEventListener('input', debouncedUpdateMeme);
  document.getElementById('bottom-text').addEventListener('input', debouncedUpdateMeme);

  // Add copy to clipboard button if supported
  if (navigator.clipboard && window.ClipboardItem) {
    const copyBtn = document.createElement('button');
    copyBtn.className = 'btn btn-secondary';
    copyBtn.innerHTML = '📋 คัดลอก';
    copyBtn.onclick = copyMemeToClipboard;

    const buttonGroup = document.querySelector('.button-group');
    if (buttonGroup) {
      buttonGroup.insertBefore(copyBtn, document.getElementById('download-btn'));
    }
  }
});
