<?php

namespace App\Handlers;

use App\Core\Database;
use App\Models\Employee;
use App\Models\LeaveRequest;
use App\Services\TelegramService;
use DateTime;

class HrSummaryHandler
{
    private TelegramService $telegramService;
    private int $chatId;
    private int $userId; // Telegram User ID of the HR person
    private ?int $hrEmployeeId = null; // System's employee ID of the HR person

    /**
     * @param TelegramService $telegramService
     * @param int $chatId
     * @param int $userId
     */
    public function __construct(TelegramService $telegramService, int $chatId, int $userId)
    {
        $this->telegramService = $telegramService;
        $this->chatId = $chatId;
        $this->userId = $userId;
        $this->loadHrEmployeeIdAndCheckPermissions();
    }

    private function loadHrEmployeeIdAndCheckPermissions(): void
    {
        $employeeModel = new Employee(Database::getInstance());
        $hrUser = $employeeModel->findByChatUserId((string) $this->userId);
        if ($hrUser && $hrUser->id) {
            // Ensure is_hr property exists and is boolean
            if (isset($hrUser->is_hr) && $hrUser->is_hr === true) {
                $this->hrEmployeeId = $hrUser->id;
                // log_message("HrSummaryHandler: HR Employee ID {$this->hrEmployeeId} loaded for Telegram User ID {$this->userId}");
            } else {
                log_message("HrSummaryHandler: User {$this->userId} (Employee ID: {$hrUser->id}) is not an HR personnel.");
                $this->telegramService->sendMessage($this->chatId, "ขออภัยค่ะ คุณไม่มีสิทธิ์ใช้คำสั่งนี้ (HR Only)");
            }
        } else {
            log_message("HrSummaryHandler: No employee record found for user {$this->userId} requesting HR summary.");
            $this->telegramService->sendMessage($this->chatId, "ขออภัยค่ะ ไม่พบข้อมูลพนักงานของคุณในระบบ");
        }
    }

    /**
     * @return null
     */
    public function showTodaysAbsences(): void
    {
        if ($this->hrEmployeeId === null) {
            // Message indicating lack of permission or user not found already sent by loadHrEmployeeIdAndCheckPermissions
            return;
        }

        $todayDateString = date('Y-m-d');
        $leaveRequestModel = new LeaveRequest(Database::getInstance());
        $absencesToday = $leaveRequestModel->getAbsencesForDate($todayDateString);

        $todayFormatted = (new DateTime($todayDateString))->format('d/m/Y');
        $responseText = "*สรุปพนักงานที่ไม่เข้าออฟฟิศวันนี้ ({$todayFormatted}):*\n\n";

        if (empty($absencesToday)) {
            $responseText .= "_ไม่พบข้อมูลพนักงานที่ลา/WFH ในวันนี้ค่ะ_";
            $this->telegramService->sendMessage($this->chatId, $responseText, null, "MarkdownV2");
            return;
        }

        $groupedAbsences = [];
        foreach ($absencesToday as $absence) {
            // $absence is a LeaveRequest object
            $typeName = ($absence->leaveType && $absence->leaveType->name) ? $absence->leaveType->name : "ไม่ทราบประเภท";
            $employeeName = ($absence->employee && $absence->employee->first_name) ? ($absence->employee->first_name." ".($absence->employee->last_name ?? '')) : "ไม่ทราบชื่อ";

            $groupedAbsences[$typeName][] = $employeeName;
        }

        if (empty($groupedAbsences)) {
            $responseText .= "_ไม่พบข้อมูลที่สามารถจัดกลุ่มได้_";
        } else {
            ksort($groupedAbsences); // Sort by leave type name for consistent order
            foreach ($groupedAbsences as $leaveTypeName => $employeeNames) {
                $responseText .= "*{$this->escapeMarkdown($leaveTypeName)}:*\n";
                if (empty($employeeNames)) {
                    // This case should ideally not happen if an entry in groupedAbsences exists
                    $responseText .= "  • _ไม่มี_\n";
                } else {
                    sort($employeeNames); // Sort employee names alphabetically
                    foreach ($employeeNames as $name) {
                        $responseText .= "  • {$this->escapeMarkdown($name)}\n";
                    }
                }
                $responseText .= "\n";
            }
        }

        $this->telegramService->sendMessage($this->chatId, trim($responseText), null, "MarkdownV2");
    }

    /**
     * @param string $text
     */
    private function escapeMarkdown(string $text): string
    {
        return preg_replace('/([_*\[\]()~`>#\+\-=|{}.!\\\\])/', '\\\\$1', $text);
    }
}
