<?php

namespace App\Models;

use App\Core\Database;
use PDO;

class LeaveType
{
    private Database $db;
    private string $table_name = "leave_types";

    // Properties
    public ?int $id = null;
    public string $name;
    public ?string $alias = null;
    public bool $requires_approval = true;
    public bool $deducts_balance = true;
    public ?string $created_at = null;

    /**
     * @param Database $db
     */
    public function __construct(?Database $db = null)
    {
        $this->db = $db ?: Database::getInstance();
    }

    // Setter methods
    /**
     * @param int $id
     * @return mixed
     */
    public function setId(int $id): self
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @param string $name
     * @return mixed
     */
    public function setName(string $name): self
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @param string $alias
     * @return mixed
     */
    public function setAlias(?string $alias): self
    {
        $this->alias = $alias;
        return $this;
    }

    /**
     * @param bool $requires_approval
     * @return mixed
     */
    public function setRequiresApproval(bool $requires_approval): self
    {
        $this->requires_approval = $requires_approval;
        return $this;
    }

    /**
     * @param bool $deducts_balance
     * @return mixed
     */
    public function setDeductsBalance(bool $deducts_balance): self
    {
        $this->deducts_balance = $deducts_balance;
        return $this;
    }

    /**
     * Create a new leave type.
     * @return bool True on success, false on failure.
     */
    public function create(): bool
    {
        $query = "INSERT INTO {$this->table_name} (name, alias, requires_approval, deducts_balance)
                  VALUES (:name, :alias, :requires_approval, :deducts_balance)";

        $this->db->query($query);
        $this->db->bind(':name', $this->name);
        $this->db->bind(':alias', $this->alias, $this->alias === null ? PDO::PARAM_NULL : PDO::PARAM_STR);
        $this->db->bind(':requires_approval', $this->requires_approval, PDO::PARAM_BOOL);
        $this->db->bind(':deducts_balance', $this->deducts_balance, PDO::PARAM_BOOL);

        if ($this->db->execute()) {
            $this->id = (int) $this->db->lastInsertId();
            return true;
        }
        return false;
    }

    /**
     * Read leave types.
     * @param ?int $id If null, fetches all. Otherwise, fetches by ID.
     * @return mixed Array of leave types, a single leave type object, or false.
     */
    public function read(?int $id = null)
    {
        if ($id !== null) {
            $query = "SELECT * FROM {$this->table_name} WHERE id = :id LIMIT 1";
            $this->db->query($query);
            $this->db->bind(':id', $id, PDO::PARAM_INT);
            $row = $this->db->single();
            if ($row) {
                $this->id = (int) $row['id'];
                $this->name = $row['name'];
                $this->alias = $row['alias'];
                $this->requires_approval = (bool) $row['requires_approval'];
                $this->deducts_balance = (bool) $row['deducts_balance'];
                $this->created_at = $row['created_at'];
                return $this;
            }
            return false;
        } else {
            $query = "SELECT * FROM {$this->table_name} ORDER BY name ASC";
            $this->db->query($query);
            return $this->db->resultSet();
        }
    }

    /**
     * Find a leave type by its alias.
     * @param string $alias The alias to search for.
     * @return self|false The LeaveType object if found, otherwise false.
     */
    public function findByAlias(string $alias)
    {
        $query = "SELECT * FROM {$this->table_name} WHERE alias = :alias LIMIT 1";
        $this->db->query($query);
        $this->db->bind(':alias', $alias);
        $row = $this->db->single();

        if ($row) {
            $this->id = (int) $row['id'];
            $this->name = $row['name'];
            $this->alias = $row['alias'];
            $this->requires_approval = (bool) $row['requires_approval'];
            $this->deducts_balance = (bool) $row['deducts_balance'];
            $this->created_at = $row['created_at'];
            return $this;
        }
        return false;
    }

    /**
     * Update an existing leave type.
     * @return bool True on success, false on failure.
     */
    public function update(): bool
    {
        if ($this->id === null) {
            return false;
        }

        $query = "UPDATE {$this->table_name} SET
                    name = :name,
                    alias = :alias,
                    requires_approval = :requires_approval,
                    deducts_balance = :deducts_balance
                  WHERE id = :id";

        $this->db->query($query);
        $this->db->bind(':id', $this->id, PDO::PARAM_INT);
        $this->db->bind(':name', $this->name);
        $this->db->bind(':alias', $this->alias, $this->alias === null ? PDO::PARAM_NULL : PDO::PARAM_STR);
        $this->db->bind(':requires_approval', $this->requires_approval, PDO::PARAM_BOOL);
        $this->db->bind(':deducts_balance', $this->deducts_balance, PDO::PARAM_BOOL);

        return $this->db->execute();
    }

    /**
     * Delete a leave type.
     * @param ?int $id If null, uses $this->id.
     * @return bool True on success, false on failure.
     */
    public function delete(?int $id = null): bool
    {
        $id_to_delete = $id ?? $this->id;
        if ($id_to_delete === null) {
            return false;
        }

        $query = "DELETE FROM {$this->table_name} WHERE id = :id";
        $this->db->query($query);
        $this->db->bind(':id', $id_to_delete, PDO::PARAM_INT);

        return $this->db->execute();
    }
}
