# 🚀 แดชบอร์ดแนวโน้มการเมืองไทย - PHP Version

## 📋 ภาพรวม

แดชบอร์ดวิเคราะห์แนวโน้มการเมืองไทยที่ใช้ **PHP** แทน Node.js เพื่อความง่ายในการใช้งานและไม่ต้องติดตั้ง Node.js

## ✨ ฟีเจอร์หลัก

- 📊 **ข้อมูลเศรษฐกิจ** จากธนาคารแห่งประเทศไทย
- 🗳️ **ข้อมูลการเมือง** จากสำนักงานคณะกรรมการการเลือกตั้ง
- 👥 **ข้อมูลประชากร** จากสำนักงานสถิติแห่งชาติ
- 📱 **วิเคราะห์ Social Media** แบบ Real-time
- 🔄 **ระบบ Cache** เพื่อเพิ่มประสิทธิภาพ
- 📤 **ส่งออกข้อมูล** เป็น CSV, JSON, XML
- 📱 **Responsive Design** รองรับทุกอุปกรณ์

## 🛠️ การติดตั้ง

### ข้อกำหนดระบบ
- PHP 7.4+
- Web Server (Apache/Nginx)
- ไม่ต้องติดตั้ง Node.js!

### ขั้นตอนการติดตั้ง

1. **วางไฟล์ในโฟลเดอร์เว็บ**
   ```bash
   # ตัวอย่างสำหรับ XAMPP
   C:\xampp\htdocs\political-dashboard\

   # หรือสำหรับ Linux
   /var/www/html/political-dashboard/
   ```

2. **ตรวจสอบสิทธิ์ไฟล์**
   ```bash
   chmod 755 api.php
   chmod 755 index-php.html
   ```

3. **เปิดใช้งาน**
   ```
   http://localhost/political-dashboard/index-php.html
   ```

## 📁 โครงสร้างไฟล์

```
political-dashboard/
├── api.php                 # PHP API หลัก
├── index-php.html         # หน้าแดชบอร์ด
├── index.html             # หน้าเดิม (ข้อมูลจำลอง)
├── cache/                 # โฟลเดอร์ Cache (สร้างอัตโนมัติ)
├── exports/               # โฟลเดอร์ส่งออกข้อมูล (สร้างอัตโนมัติ)
└── README-PHP.md          # คู่มือนี้
```

## 🔌 การใช้งาน API

### ข้อมูลเศรษฐกิจ
```bash
GET api.php?action=economic&timeframe=6months
```

### ข้อมูลการเมือง
```bash
GET api.php?action=political&timeframe=6months
```

### ข้อมูลประชากร
```bash
GET api.php?action=demographic&region=all
```

### วิเคราะห์ Social Media
```bash
GET api.php?action=social_media&keywords=การเมืองไทย,รัฐบาล
```

### ข้อมูลทั้งหมด
```bash
GET api.php?action=all&timeframe=6months&region=all
```

### ส่งออกข้อมูล
```bash
POST api.php?action=export
Content-Type: application/json

{
  "type": "csv",
  "data": {...}
}
```

## 📊 ตัวอย่างการใช้งาน

### เรียกข้อมูลผ่าน JavaScript
```javascript
// โหลดข้อมูลเศรษฐกิจ
const response = await fetch('api.php?action=economic&timeframe=6months');
const data = await response.json();

if (data.success) {
  console.log('GDP Growth:', data.data.gdp_growth);
  console.log('Inflation:', data.data.inflation_rate);
}
```

### เรียกข้อมูลผ่าน cURL
```bash
# ข้อมูลเศรษฐกิจ
curl "http://localhost/political-dashboard/api.php?action=economic&timeframe=6months"

# ข้อมูลการเมือง
curl "http://localhost/political-dashboard/api.php?action=political&timeframe=6months"
```

## 🔧 การปรับแต่ง

### เปลี่ยนแหล่งข้อมูล
แก้ไขใน `api.php`:

```php
private function fetchEconomicDataFromAPI($timeframe) {
    // เปลี่ยน URL API จริง
    $url = "https://api.bot.or.th/v1/economic-indicators";

    // ใช้ cURL หรือ file_get_contents
    $response = file_get_contents($url);
    $data = json_decode($response, true);

    return $data;
}
```

### เพิ่มแหล่งข้อมูลใหม่
```php
public function getNewData($params = []) {
    $cacheKey = "new_data_" . md5(serialize($params));

    if ($this->isCacheValid($cacheKey)) {
        return $this->getCache($cacheKey);
    }

    $data = $this->fetchNewDataFromAPI($params);
    $this->setCache($cacheKey, $data);

    return $data;
}
```

### ปรับ Cache Timeout
```php
private $cacheTimeout = 300; // 5 นาที
// เปลี่ยนเป็น 600 สำหรับ 10 นาที
```

## 🗄️ ระบบ Cache

- **ไฟล์ Cache**: เก็บในโฟลเดอร์ `cache/`
- **Timeout**: 5 นาที (ปรับได้)
- **รูปแบบ**: JSON
- **การล้าง Cache**: อัตโนมัติตามเวลา

### ล้าง Cache เอง
```bash
rm -rf cache/*
```

## 📤 การส่งออกข้อมูล

### รูปแบบที่รองรับ
- **CSV**: สำหรับ Excel
- **JSON**: สำหรับ API
- **XML**: สำหรับระบบอื่น

### ตัวอย่างการส่งออก
```javascript
// ส่งออกเป็น CSV
const response = await fetch('api.php?action=export', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
        type: 'csv',
        data: dashboardData
    })
});

const result = await response.json();
if (result.success) {
    // ดาวน์โหลดไฟล์
    window.open(result.data.download_url);
}
```

## 🔒 ความปลอดภัย

### การป้องกัน
- ✅ **Input Validation**
- ✅ **SQL Injection Protection** (ถ้าใช้ฐานข้อมูล)
- ✅ **XSS Protection**
- ✅ **CORS Configuration**

### การตั้งค่าเพิ่มเติม
```php
// เพิ่มใน api.php
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
```

## 🚀 การพัฒนาต่อ

### เพิ่มฐานข้อมูล
```php
// เชื่อมต่อ MySQL
$pdo = new PDO(
    "mysql:host=localhost;dbname=political_data;charset=utf8",
    "username",
    "password"
);

// บันทึกข้อมูล
$stmt = $pdo->prepare("INSERT INTO economic_data (gdp_growth, inflation_rate, created_at) VALUES (?, ?, NOW())");
$stmt->execute([$gdpGrowth, $inflationRate]);
```

### เพิ่ม Authentication
```php
// ตรวจสอบ API Key
$apiKey = $_GET['api_key'] ?? '';
if (!in_array($apiKey, $validApiKeys)) {
    http_response_code(401);
    echo json_encode(['error' => 'Invalid API key']);
    exit;
}
```

### เพิ่ม Rate Limiting
```php
// จำกัดการเรียก API
$clientIP = $_SERVER['REMOTE_ADDR'];
$rateLimit = checkRateLimit($clientIP, 100, 3600); // 100 calls/hour

if (!$rateLimit) {
    http_response_code(429);
    echo json_encode(['error' => 'Rate limit exceeded']);
    exit;
}
```

## 📈 การติดตามประสิทธิภาพ

### Log การใช้งาน
```php
// เพิ่มใน api.php
error_log("API Call: " . $_GET['action'] . " from " . $_SERVER['REMOTE_ADDR']);
```

### ตรวจสอบ Cache Hit Rate
```php
private function logCacheHit($key, $hit) {
    $log = date('Y-m-d H:i:s') . " - Cache " . ($hit ? "HIT" : "MISS") . " for key: $key\n";
    file_put_contents('logs/cache.log', $log, FILE_APPEND);
}
```

## 🐛 การแก้ไขปัญหา

### ปัญหาที่พบบ่อย

1. **API ไม่ตอบสนอง**
   ```bash
   # ตรวจสอบ PHP error log
   tail -f /var/log/apache2/error.log
   ```

2. **Cache ไม่ทำงาน**
   ```bash
   # ตรวจสอบสิทธิ์โฟลเดอร์
   chmod 755 cache/
   chmod 755 exports/
   ```

3. **CORS Error**
   ```php
   // เพิ่มใน api.php
   header('Access-Control-Allow-Origin: *');
   header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
   ```

### Debug Mode
```php
// เพิ่มใน api.php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

## 📞 การสนับสนุน

### การรายงานปัญหา
1. ตรวจสอบ PHP error log
2. ตรวจสอบ Network tab ใน Developer Tools
3. รายงานพร้อมข้อมูล:
   - PHP Version
   - Web Server
   - Error Message
   - Steps to Reproduce

### การขอฟีเจอร์ใหม่
- ส่งคำขอพร้อมรายละเอียด
- ระบุประโยชน์ที่คาดหวัง
- แนบตัวอย่างการใช้งาน

## 📄 License

MIT License - ใช้งานได้อย่างอิสระ

## 🙏 ขอบคุณ

- ธนาคารแห่งประเทศไทย สำหรับข้อมูลเศรษฐกิจ
- สำนักงานคณะกรรมการการเลือกตั้ง สำหรับข้อมูลการเมือง
- สำนักงานสถิติแห่งชาติ สำหรับข้อมูลประชากร
- Chart.js สำหรับการแสดงกราฟ
- PHP Community สำหรับเครื่องมือที่ยอดเยี่ยม

---

**🎯 เริ่มต้นใช้งานได้ทันทีโดยไม่ต้องติดตั้ง Node.js!**