<?php

namespace App\Middleware;

use App\Core\Auth;
use App\Core\Database;

/**
 * Authentication Middleware
 * Validates JWT tokens and ensures user is authenticated
 */
class AuthMiddleware
{
    /**
     * @var mixed
     */
    private $auth;

    public function __construct()
    {
        $db = Database::getInstance();
        $this->auth = new Auth($db, $_ENV['JWT_SECRET'] ?? 'default-secret');
    }

    /**
     * Handle the middleware
     */
    public function handle(): bool
    {
        $token = $this->getBearerToken();

        if (!$token) {
            $this->sendUnauthorizedResponse('Authentication required');
            return false;
        }

        $user = $this->auth->validateToken($token);

        if (!$user) {
            $this->sendUnauthorizedResponse('Invalid or expired token');
            return false;
        }

        // Store user in global state for controllers to access
        $_SESSION['authenticated_user'] = $user;

        return true;
    }

    /**
     * Get Bearer token from Authorization header
     */
    private function getBearerToken(): ?string
    {
        $headers = $this->getAuthorizationHeader();

        if (!$headers) {
            return null;
        }

        if (preg_match('/Bearer\s+(.*)$/i', $headers, $matches)) {
            return $matches[1];
        }

        return null;
    }

    /**
     * Get Authorization header
     */
    private function getAuthorizationHeader(): ?string
    {
        $headers = null;

        // Try different server variables for Authorization header
        if (isset($_SERVER['Authorization'])) {
            $headers = trim($_SERVER['Authorization']);
        } elseif (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $headers = trim($_SERVER['HTTP_AUTHORIZATION']);
        } elseif (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            // For CGI/FastCGI environments
            $headers = trim($_SERVER['REDIRECT_HTTP_AUTHORIZATION']);
        } elseif (function_exists('apache_request_headers')) {
            $requestHeaders = apache_request_headers();
            if ($requestHeaders) {
                $requestHeaders = array_combine(
                    array_map('ucwords', array_keys($requestHeaders)),
                    array_values($requestHeaders)
                );

                if (isset($requestHeaders['Authorization'])) {
                    $headers = trim($requestHeaders['Authorization']);
                }
            }
        }

        return $headers;
    }

    /**
     * Send unauthorized response
     */
    private function sendUnauthorizedResponse(string $message): void
    {
        http_response_code(401);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'error' => 'Unauthorized',
            'message' => $message
        ]);
    }
}
