<?php

namespace App\Controllers;

use Exception;

class SettingsController extends BaseController
{
    /**
     * @var mixed
     */
    private $settingsFile;

    public function __construct()
    {
        parent::__construct();
        $this->settingsFile = dirname(__DIR__, 2).'/storage/settings.json';

        // Create settings file if not exists
        if (!file_exists($this->settingsFile)) {
            $this->createDefaultSettings();
        }
    }

    /**
     * Get all settings
     */
    public function getSettings()
    {
        try {
            if (!file_exists($this->settingsFile)) {
                $this->createDefaultSettings();
            }

            $settings = json_decode(file_get_contents($this->settingsFile), true);

            // Remove sensitive data from response
            if (isset($settings['system']['jwtSecret'])) {
                $settings['system']['jwtSecret'] = str_repeat('*', 20);
            }

            return $this->jsonResponse([
                'success' => true,
                'data' => $settings
            ]);

        } catch (Exception $e) {
            return $this->jsonResponse([
                'success' => false,
                'message' => 'ไม่สามารถอ่านการตั้งค่าได้'
            ], 500);
        }
    }

    /**
     * Update settings
     */
    public function updateSettings()
    {
        try {
            // Check if user is admin
            if (!$this->isAdmin()) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ไม่มีสิทธิ์ในการแก้ไขการตั้งค่า'
                ], 403);
            }

            $input = $this->getJsonInput();

            // Validate input
            if (empty($input)) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ข้อมูลการตั้งค่าไม่ถูกต้อง'
                ], 400);
            }

            // Validate required fields
            $errors = $this->validateSettings($input);
            if (!empty($errors)) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ข้อมูลไม่ถูกต้อง',
                    'errors' => $errors
                ], 400);
            }

            // Load current settings
            $currentSettings = json_decode(file_get_contents($this->settingsFile), true);

            // Merge settings
            $newSettings = array_merge_recursive($currentSettings, $input);

            // Add metadata
            $newSettings['meta'] = [
                'updated_at' => date('Y-m-d H:i:s'),
                'updated_by' => $this->getCurrentUserId(),
                'version' => $currentSettings['meta']['version'] ?? 1.0
            ];

            // Save settings
            if (file_put_contents($this->settingsFile, json_encode($newSettings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {

                // Log settings change
                $this->logSettingsChange($input);

                return $this->jsonResponse([
                    'success' => true,
                    'message' => 'บันทึกการตั้งค่าเรียบร้อยแล้ว'
                ]);
            } else {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ไม่สามารถบันทึกการตั้งค่าได้'
                ], 500);
            }

        } catch (Exception $e) {
            error_log("Settings update error: ".$e->getMessage());

            return $this->jsonResponse([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการบันทึกการตั้งค่า'
            ], 500);
        }
    }

    /**
     * Reset settings to default
     */
    public function resetSettings()
    {
        try {
            // Check if user is admin
            if (!$this->isAdmin()) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ไม่มีสิทธิ์ในการรีเซ็ตการตั้งค่า'
                ], 403);
            }

            $this->createDefaultSettings();

            return $this->jsonResponse([
                'success' => true,
                'message' => 'รีเซ็ตการตั้งค่าเรียบร้อยแล้ว'
            ]);

        } catch (Exception $e) {
            return $this->jsonResponse([
                'success' => false,
                'message' => 'ไม่สามารถรีเซ็ตการตั้งค่าได้'
            ], 500);
        }
    }

    /**
     * Export settings
     */
    public function exportSettings()
    {
        try {
            // Check if user is admin
            if (!$this->isAdmin()) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ไม่มีสิทธิ์ในการส่งออกการตั้งค่า'
                ], 403);
            }

            $settings = json_decode(file_get_contents($this->settingsFile), true);

            // Remove sensitive data
            unset($settings['system']['jwtSecret']);

            $filename = 'store_settings_'.date('Y-m-d_H-i-s').'.json';

            header('Content-Type: application/json');
            header('Content-Disposition: attachment; filename='.$filename);

            echo json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            exit;

        } catch (Exception $e) {
            return $this->jsonResponse([
                'success' => false,
                'message' => 'ไม่สามารถส่งออกการตั้งค่าได้'
            ], 500);
        }
    }

    /**
     * Import settings
     */
    public function importSettings()
    {
        try {
            // Check if user is admin
            if (!$this->isAdmin()) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ไม่มีสิทธิ์ในการนำเข้าการตั้งค่า'
                ], 403);
            }

            if (!isset($_FILES['settings_file'])) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'กรุณาเลือกไฟล์การตั้งค่า'
                ], 400);
            }

            $file = $_FILES['settings_file'];

            if ($file['error'] !== UPLOAD_ERR_OK) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'เกิดข้อผิดพลาดในการอัพโลดไฟล์'
                ], 400);
            }

            $content = file_get_contents($file['tmp_name']);
            $importedSettings = json_decode($content, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ไฟล์การตั้งค่าไม่ถูกต้อง'
                ], 400);
            }

            // Validate imported settings
            $errors = $this->validateSettings($importedSettings);
            if (!empty($errors)) {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ข้อมูลในไฟล์ไม่ถูกต้อง',
                    'errors' => $errors
                ], 400);
            }

            // Add metadata
            $importedSettings['meta'] = [
                'updated_at' => date('Y-m-d H:i:s'),
                'updated_by' => $this->getCurrentUserId(),
                'imported_at' => date('Y-m-d H:i:s'),
                'version' => 1.0
            ];

            // Save settings
            if (file_put_contents($this->settingsFile, json_encode($importedSettings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                return $this->jsonResponse([
                    'success' => true,
                    'message' => 'นำเข้าการตั้งค่าเรียบร้อยแล้ว'
                ]);
            } else {
                return $this->jsonResponse([
                    'success' => false,
                    'message' => 'ไม่สามารถบันทึกการตั้งค่าได้'
                ], 500);
            }

        } catch (Exception $e) {
            return $this->jsonResponse([
                'success' => false,
                'message' => 'ไม่สามารถนำเข้าการตั้งค่าได้'
            ], 500);
        }
    }

    /**
     * Create default settings
     */
    private function createDefaultSettings()
    {
        $defaultSettings = [
            'general' => [
                'storeName' => 'Fashion Store',
                'storeDescription' => 'เสื้อผ้าแฟชั่นคุณภาพดี ราคาย่อมเยา ส่งถึงบ้าน',
                'storePhone' => '081-234-5678',
                'storeEmail' => 'fashionstore@email.com',
                'storeAddress' => '123 ถนนสุขุมวิท กรุงเทพฯ 10110',
                'storeFacebook' => 'https://facebook.com/fashionstore',
                'storeLine' => '',
                'logoUrl' => ''
            ],
            'payment' => [
                'promptPayNumber' => '0812345678',
                'promptPayName' => 'Fashion Store',
                'enableCOD' => false,
                'bankAccounts' => []
            ],
            'shipping' => [
                'defaultShippingFee' => 50,
                'freeShippingMin' => 1000,
                'shippingDaysMin' => 2,
                'shippingDaysMax' => 5,
                'shippingZones' => [
                    [
                        'name' => 'กรุงเทพและปริมณฑ์',
                        'fee' => 50,
                        'provinces' => ['กรุงเทพมหานคร', 'นครปฐม', 'นนทบุรี', 'ปทุมธานี', 'สมุทรปราการ', 'สมุทรสาคร']
                    ]
                ]
            ],
            'products' => [
                'categories' => ['เสื้อ', 'กางเกง', 'กระโปรง', 'เดรส', 'เสื้อนอก'],
                'sizes' => ['XS', 'S', 'M', 'L', 'XL', 'XXL'],
                'colors' => [
                    ['name' => 'ขาว', 'hex' => '#ffffff'],
                    ['name' => 'ดำ', 'hex' => '#000000'],
                    ['name' => 'แดง', 'hex' => '#ff0000'],
                    ['name' => 'น้ำเงิน', 'hex' => '#0000ff']
                ]
            ],
            'notifications' => [
                'adminEmail' => 'admin@fashionstore.com',
                'notifyNewOrder' => true,
                'notifyPaymentReceived' => true,
                'notifyLowStock' => true,
                'sendOrderConfirmation' => true,
                'sendShippingNotification' => true,
                'lineNotifyToken' => ''
            ],
            'system' => [
                'language' => 'th',
                'currency' => 'THB',
                'timezone' => 'Asia/Bangkok',
                'apiBaseUrl' => 'http://localhost/store/api',
                'enableDebugMode' => false,
                'enableMaintenanceMode' => false,
                'jwtSecret' => $this->generateRandomKey(),
                'sessionTimeout' => 60
            ],
            'meta' => [
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
                'version' => 1.0
            ]
        ];

        // Create storage directory if not exists
        $storageDir = dirname($this->settingsFile);
        if (!is_dir($storageDir)) {
            mkdir($storageDir, 0755, true);
        }

        file_put_contents($this->settingsFile, json_encode($defaultSettings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }

    /**
     * Validate settings data
     */
    private function validateSettings($settings)
    {
        $errors = [];

        // Validate general settings
        if (isset($settings['general'])) {
            if (empty($settings['general']['storeName'])) {
                $errors[] = 'ชื่อร้านค้าไม่สามารถเว้นว่างได้';
            }

            if (!empty($settings['general']['storeEmail']) && !filter_var($settings['general']['storeEmail'], FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'รูปแบบอีเมลไม่ถูกต้อง';
            }
        }

        // Validate payment settings
        if (isset($settings['payment'])) {
            if (!empty($settings['payment']['promptPayNumber']) && !preg_match('/^[0-9]{10,13}$/', $settings['payment']['promptPayNumber'])) {
                $errors[] = 'หมายเลข PromptPay ไม่ถูกต้อง';
            }
        }

        // Validate shipping settings
        if (isset($settings['shipping'])) {
            if (isset($settings['shipping']['defaultShippingFee']) && $settings['shipping']['defaultShippingFee'] < 0) {
                $errors[] = 'ค่าจัดส่งต้องมากกว่าหรือเท่ากับ 0';
            }
        }

        // Validate system settings
        if (isset($settings['system'])) {
            if (!empty($settings['system']['apiBaseUrl']) && !filter_var($settings['system']['apiBaseUrl'], FILTER_VALIDATE_URL)) {
                $errors[] = 'URL API ไม่ถูกต้อง';
            }
        }

        return $errors;
    }

    /**
     * Log settings change
     */
    private function logSettingsChange($changes)
    {
        $logData = [
            'timestamp' => date('Y-m-d H:i:s'),
            'user_id' => $this->getCurrentUserId(),
            'action' => 'settings_update',
            'changes' => $changes,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ];

        $logFile = dirname(__DIR__, 2).'/storage/logs/settings.log';
        $logDir = dirname($logFile);

        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }

        file_put_contents($logFile, json_encode($logData)."\n", FILE_APPEND | LOCK_EX);
    }

    /**
     * Generate random key
     */
    private function generateRandomKey($length = 64)
    {
        $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*()_+-=[]{}|;:,.<>?';
        $charactersLength = strlen($characters);
        $randomString = '';

        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }

        return $randomString;
    }

    /**
     * Check if current user is admin
     */
    private function isAdmin()
    {
        // This should be implemented based on your authentication system
        // For now, we'll assume it's always true for demonstration
        return true;
    }

    /**
     * Get current user ID
     */
    private function getCurrentUserId()
    {
        // This should be implemented based on your authentication system
        return 1;
    }
}
