<?php

namespace App\Middleware;

use App\Core\Validator;

/**
 * Validation Middleware
 * Provides request validation functionality
 */
class ValidationMiddleware
{
    /**
     * @var mixed
     */
    private $validator;

    public function __construct()
    {
        $this->validator = new Validator();
    }

    /**
     * Handle the request
     */
    public function handle($request = null, $next = null, $rules = [])
    {
        // If called without parameters (from Router), just continue
        if ($request === null && $next === null) {
            return true;
        }

        // If no rules provided, continue to next middleware
        if (empty($rules)) {
            return $next($request);
        }

        // Get request data
        $data = $this->getRequestData($request);

        // Validate the data
        $isValid = $this->validator->validate($data, $rules);

        if (!$isValid) {
            // Return validation error response
            http_response_code(400);
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'error' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ]);
            return;
        }

        // Continue to next middleware
        return $next($request);
    }

    /**
     * Get request data based on content type
     */
    private function getRequestData($request)
    {
        $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

        if ($method === 'GET') {
            return $_GET;
        }

        $contentType = $_SERVER['CONTENT_TYPE'] ?? '';

        if (strpos($contentType, 'application/json') !== false) {
            $input = file_get_contents('php://input');
            return json_decode($input, true) ?: [];
        }

        if ($method === 'POST') {
            return $_POST;
        }

        return [];
    }

    /**
     * Static method for easy validation in controllers
     */
    public static function validate(array $data, array $rules): array
    {
        $validator = new Validator();
        $isValid = $validator->validate($data, $rules);

        return [
            'valid' => $isValid,
            'errors' => $validator->getErrors()
        ];
    }
}
