# Kotchasan PHP Framework

## Introduction

Kotchasan is a lightweight and high-performance PHP Framework designed for rapid and secure web application development.

## Key Features

- 🚀 **High Performance** - Built for speed and efficiency
- 🔒 **Security** - Built-in security features
- 📦 **Modular** - Flexible and extensible module system
- 🎨 **MVC Pattern** - Model-View-Controller support
- 🌐 **Multi-language** - Multi-language support
- 📱 **Responsive** - Mobile-first design

## Code Example

```php
<?php
namespace App\Controllers;

use Kotchasan\Controller;
use Kotchasan\Http\Request;

class HomeController extends Controller
{
    public function index(Request $request)
    {
        $data = [
            'title' => 'Welcome to Kotchasan',
            'message' => 'High-performance framework'
        ];

        return $this->view('home', $data);
    }
}
```

## Installation

### System Requirements

- PHP 8.0 or higher
- Composer
- Web Server (Apache/Nginx)
- MySQL 5.7 or higher

### Installation Steps

1. **Clone Repository**
   ```bash
   git clone https://github.com/your-org/kotchasan.git
   cd kotchasan
   ```

2. **Install Dependencies**
   ```bash
   composer install
   ```

3. **Setup Environment**
   ```bash
   cp .env.example .env
   # Edit .env file with your configuration
   ```

4. **Setup Database**
   ```bash
   php artisan migrate
   php artisan db:seed
   ```

5. **Start Development Server**
   ```bash
   php artisan serve
   ```

## Getting Started

### Creating a New Controller

```php
<?php
namespace App\Controllers;

use Kotchasan\Controller;

class ApiController extends Controller
{
    public function users()
    {
        $users = User::all();
        return $this->json($users);
    }
}
```

### Creating a Model

```php
<?php
namespace App\Models;

use Kotchasan\Model;

class User extends Model
{
    protected $table = 'users';

    protected $fillable = [
        'name', 'email', 'password'
    ];

    protected $hidden = [
        'password'
    ];
}
```

### Creating a View

```html
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $title ?></title>
</head>
<body>
    <h1><?= $title ?></h1>
    <p><?= $message ?></p>
</body>
</html>
```

## Routing Configuration

```php
<?php
// routes/web.php

use App\Controllers\HomeController;
use App\Controllers\ApiController;

// Web Routes
Route::get('/', [HomeController::class, 'index']);
Route::get('/about', [HomeController::class, 'about']);

// API Routes
Route::prefix('api')->group(function () {
    Route::get('/users', [ApiController::class, 'users']);
    Route::post('/users', [ApiController::class, 'store']);
});
```

## Security Features

### CSRF Protection

```php
// In forms
<form method="POST">
    <?= csrf_token() ?>
    <!-- other fields -->
</form>

// Validate in Controller
public function store(Request $request)
{
    if (!$request->validateCsrf()) {
        return $this->error('Invalid CSRF token');
    }
    // process data
}
```

### Password Hashing

```php
// Hash password
$hashedPassword = password_hash($password, PASSWORD_DEFAULT);

// Verify password
if (password_verify($password, $hashedPassword)) {
    // password is correct
}
```

## Debugging and Logging

```php
use Kotchasan\Log\Logger;

// Log messages
Logger::info('User logged in', ['user_id' => $userId]);
Logger::error('Database connection failed', ['error' => $error]);

// Debug mode
if (config('app.debug')) {
    dump($variable);
    dd($data); // dump and die
}
```

## Testing

### Unit Testing

```php
<?php
namespace Tests\Unit;

use PHPUnit\Framework\TestCase;
use App\Models\User;

class UserTest extends TestCase
{
    public function testUserCreation()
    {
        $user = new User([
            'name' => 'John Doe',
            'email' => 'john@example.com'
        ]);

        $this->assertEquals('John Doe', $user->name);
        $this->assertEquals('john@example.com', $user->email);
    }
}
```

## Deployment

### Production Setup

1. **Environment Configuration**
   ```bash
   # Set APP_ENV=production
   # Set APP_DEBUG=false
   ```

2. **Optimize Application**
   ```bash
   php artisan optimize
   php artisan config:cache
   php artisan route:cache
   ```

3. **Web Server Configuration**
   ```nginx
   server {
       listen 80;
       server_name your-domain.com;
       root /path/to/kotchasan/public;

       location / {
           try_files $uri $uri/ /index.php?$query_string;
       }

       location ~ \.php$ {
           fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
           fastcgi_index index.php;
           include fastcgi_params;
       }
   }
   ```

## Additional Resources

- 📚 [Getting Started Guide](getting-started.md)
- 🔧 [Installation Guide](installation.md)
- 📖 [API Reference](api-reference.md)
- 🐛 [Troubleshooting](troubleshooting.md)
- 💬 [Community Forum](https://forum.kotchasan.com)

## Support

If you encounter issues or need help:

- 📧 Email: support@kotchasan.com
- 💬 Discord: [Kotchasan Community](https://discord.gg/kotchasan)
- 🐛 GitHub Issues: [Report Bug](https://github.com/kotchasan/framework/issues)

---

**Kotchasan Framework** - Build fast and secure web applications

