# API Reference - Kotchasan Framework

## Controller

### Base Controller

```php
<?php
namespace App\Controllers;

use Kotchasan\Controller;
use Kotchasan\Http\Request;
use Kotchasan\Http\Response;

class BaseController extends Controller
{
    protected $middleware = [];

    public function __construct()
    {
        $this->middleware($this->middleware);
    }

    protected function middleware($middleware)
    {
        // Apply middleware
    }
}
```

### Controller Methods

#### `view($template, $data = [])`
Render view template

```php
public function index()
{
    return $this->view('home', [
        'title' => 'Home Page',
        'users' => User::all()
    ]);
}
```

#### `json($data, $status = 200)`
Return JSON response

```php
public function api()
{
    return $this->json([
        'status' => 'success',
        'data' => User::all()
    ]);
}
```

#### `redirect($url, $status = 302)`
Redirect to URL

```php
public function store(Request $request)
{
    // Process data
    return $this->redirect('/success');
}
```

## Request

### Request Methods

#### `get($key, $default = null)`
Get GET parameter

```php
$id = $request->get('id');
$name = $request->get('name', 'Guest');
```

#### `post($key, $default = null)`
Get POST data

```php
$title = $request->post('title');
$content = $request->post('content');
```

#### `input($key, $default = null)`
Get input from GET or POST

```php
$search = $request->input('search');
```

#### `file($key)`
Get uploaded file

```php
$file = $request->file('avatar');
if ($file) {
    $filename = $file->store('uploads');
}
```

#### `has($key)`
Check if key exists

```php
if ($request->has('email')) {
    // Process email
}
```

#### `validate($rules)`
Validate input data

```php
$request->validate([
    'name' => 'required|string|max:255',
    'email' => 'required|email|unique:users',
    'password' => 'required|min:8'
]);
```

### Request Properties

```php
// URL information
$request->url();           // Full URL
$request->path();         // Path only
$request->method();       // HTTP method
$request->isAjax();       // Is AJAX request
$request->isJson();       // Is JSON request

// Headers
$request->header('Content-Type');
$request->userAgent();
$request->ip();

// Session
$request->session();
$request->cookie('theme');
```

## Response

### Response Methods

#### `json($data, $status = 200)`
Return JSON response

```php
return response()->json([
    'message' => 'Success',
    'data' => $data
], 200);
```

#### `view($template, $data = [])`
Render view

```php
return response()->view('users.index', [
    'users' => $users
]);
```

#### `redirect($url, $status = 302)`
Redirect to URL

```php
return response()->redirect('/dashboard');
```

#### `download($file, $name = null)`
Download file

```php
return response()->download($filePath, 'report.pdf');
```

#### `stream($callback, $headers = [])`
Stream response

```php
return response()->stream(function () {
    echo "Data chunk 1\n";
    flush();
    sleep(1);
    echo "Data chunk 2\n";
});
```

## Model

### Base Model

```php
<?php
namespace App\Models;

use Kotchasan\Model;

class User extends Model
{
    protected $table = 'users';
    protected $primaryKey = 'id';
    protected $fillable = ['name', 'email', 'password'];
    protected $hidden = ['password'];
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed'
    ];
}
```

### Model Methods

#### `find($id)`
Find by ID

```php
$user = User::find(1);
$user = User::findOrFail(1); // Throw exception if not found
```

#### `where($column, $operator, $value)`
Add where condition

```php
$users = User::where('status', 'active')->get();
$users = User::where('age', '>', 18)->get();
$users = User::where('name', 'like', '%john%')->get();
```

#### `create($data)`
Create new record

```php
$user = User::create([
    'name' => 'John Doe',
    'email' => 'john@example.com',
    'password' => 'password123'
]);
```

#### `update($data)`
Update record

```php
$user = User::find(1);
$user->update(['name' => 'Jane Doe']);
```

#### `delete()`
Delete record

```php
$user = User::find(1);
$user->delete();
```

### Relationships

#### `hasMany($related, $foreignKey, $localKey)`
One-to-Many relationship

```php
public function posts()
{
    return $this->hasMany(Post::class);
}

// Usage
$user = User::find(1);
$posts = $user->posts;
```

#### `belongsTo($related, $foreignKey, $ownerKey)`
Many-to-One relationship

```php
public function user()
{
    return $this->belongsTo(User::class);
}

// Usage
$post = Post::find(1);
$user = $post->user;
```

#### `belongsToMany($related, $table, $foreignPivotKey, $relatedPivotKey)`
Many-to-Many relationship

```php
public function roles()
{
    return $this->belongsToMany(Role::class);
}

// Usage
$user = User::find(1);
$roles = $user->roles;
```

## Database

### Query Builder

```php
use Kotchasan\Database\DB;

// Select
$users = DB::table('users')->get();
$user = DB::table('users')->where('id', 1)->first();

// Insert
DB::table('users')->insert([
    'name' => 'John Doe',
    'email' => 'john@example.com'
]);

// Update
DB::table('users')
    ->where('id', 1)
    ->update(['name' => 'Jane Doe']);

// Delete
DB::table('users')->where('id', 1)->delete();
```

### Raw Queries

```php
// Raw SQL
$users = DB::select('SELECT * FROM users WHERE status = ?', ['active']);

// Raw insert
DB::insert('INSERT INTO users (name, email) VALUES (?, ?)', ['John', 'john@example.com']);

// Raw update
DB::update('UPDATE users SET name = ? WHERE id = ?', ['Jane', 1]);
```

## Validation

### Validation Rules

```php
use Kotchasan\Validation\Validator;

$validator = Validator::make($data, [
    'name' => 'required|string|max:255',
    'email' => 'required|email|unique:users',
    'password' => 'required|min:8|confirmed',
    'age' => 'required|integer|min:18|max:100',
    'website' => 'nullable|url',
    'avatar' => 'nullable|image|max:2048'
]);
```

### Custom Validation

```php
use Kotchasan\Validation\Rule;

Validator::extend('phone', function ($attribute, $value, $parameters, $validator) {
    return preg_match('/^[0-9]{10}$/', $value);
});

// Usage
$validator = Validator::make($data, [
    'phone' => 'required|phone'
]);
```

## Cache

### Cache Methods

```php
use Kotchasan\Cache\Cache;

// Store
Cache::put('key', 'value', 3600); // 1 hour
Cache::put('user:1', $user, 3600);

// Retrieve
$value = Cache::get('key');
$user = Cache::get('user:1');

// Store if not exists
Cache::add('key', 'value', 3600);

// Forever
Cache::forever('key', 'value');

// Remove
Cache::forget('key');
Cache::flush(); // Remove all
```

### Cache Tags

```php
// Store with tags
Cache::tags(['users', 'posts'])->put('user_posts', $data, 3600);

// Remove by tags
Cache::tags(['users'])->flush();
```

## Session

### Session Methods

```php
use Kotchasan\Session\Session;

// Store
Session::put('key', 'value');
Session::put('user', $user);

// Retrieve
$value = Session::get('key');
$user = Session::get('user');

// Check
if (Session::has('user')) {
    // User is logged in
}

// Remove
Session::forget('key');
Session::flush(); // Remove all
```

## Mail

### Sending Mail

```php
use Kotchasan\Mail\Mail;

// Simple mail
Mail::send('emails.welcome', ['user' => $user], function ($message) {
    $message->to('user@example.com')
            ->subject('Welcome!');
});

// Mail with attachment
Mail::send('emails.report', $data, function ($message) {
    $message->to('admin@example.com')
            ->subject('Monthly Report')
            ->attach('/path/to/report.pdf');
});
```

### Mail Templates

```html
<!-- resources/views/emails/welcome.blade.php -->
<h1>Welcome {{ $user->name }}!</h1>
<p>Thank you for joining our platform.</p>
```

## Queue

### Creating Jobs

```php
<?php
namespace App\Jobs;

use Kotchasan\Queue\Job;

class SendEmailJob extends Job
{
    protected $email;
    protected $subject;
    protected $message;

    public function __construct($email, $subject, $message)
    {
        $this->email = $email;
        $this->subject = $subject;
        $this->message = $message;
    }

    public function handle()
    {
        // Send email logic
        mail($this->email, $this->subject, $this->message);
    }
}
```

### Dispatching Jobs

```php
// Dispatch immediately
SendEmailJob::dispatch($email, $subject, $message);

// Dispatch with delay
SendEmailJob::dispatch($email, $subject, $message)
    ->delay(now()->addMinutes(5));

// Dispatch to specific queue
SendEmailJob::dispatch($email, $subject, $message)
    ->onQueue('emails');
```

## Middleware

### Creating Middleware

```php
<?php
namespace App\Middleware;

use Kotchasan\Http\Request;

class AuthMiddleware
{
    public function handle(Request $request, \Closure $next)
    {
        if (!auth()->check()) {
            return redirect('/login');
        }

        return $next($request);
    }
}
```

### Registering Middleware

```php
// In config/middleware.php
return [
    'auth' => \App\Middleware\AuthMiddleware::class,
    'guest' => \App\Middleware\GuestMiddleware::class,
    'throttle' => \App\Middleware\ThrottleMiddleware::class,
];
```

## Helpers

### Array Helpers

```php
use Kotchasan\Helpers\Arr;

$array = ['a', 'b', 'c'];

// Get value with default
$value = Arr::get($array, 'key', 'default');

// Set value
Arr::set($array, 'key', 'value');

// Check if key exists
if (Arr::has($array, 'key')) {
    // Key exists
}
```

### String Helpers

```php
use Kotchasan\Helpers\Str;

// Generate random string
$random = Str::random(32);

// Convert to snake_case
$snake = Str::snake('HelloWorld'); // hello_world

// Convert to camelCase
$camel = Str::camel('hello_world'); // helloWorld

// Generate slug
$slug = Str::slug('Hello World'); // hello-world
```

### Date Helpers

```php
use Kotchasan\Helpers\Date;

// Format date
$formatted = Date::format('Y-m-d H:i:s');

// Parse date
$date = Date::parse('2024-01-01');

// Add time
$future = Date::addDays(7);
$future = Date::addMonths(1);
```

## Error Handling

### Custom Exceptions

```php
<?php
namespace App\Exceptions;

use Exception;

class CustomException extends Exception
{
    protected $message = 'Custom error message';
    protected $code = 500;
}
```

### Exception Handling

```php
try {
    // Risky operation
    $result = riskyOperation();
} catch (CustomException $e) {
    // Handle custom exception
    return response()->json(['error' => $e->getMessage()], 500);
} catch (Exception $e) {
    // Handle general exception
    return response()->json(['error' => 'Something went wrong'], 500);
}
```

---

**Note**: This is a basic API reference. For more information, please refer to the [Getting Started Guide](getting-started.md)

