# Kotchasan PHP Framework

## แนะนำ

Kotchasan เป็น PHP Framework ที่เบาและมีประสิทธิภาพสูง ออกแบบมาเพื่อการพัฒนาเว็บแอปพลิเคชันที่รวดเร็วและปลอดภัย

## คุณสมบัติหลัก

- 🚀 **ประสิทธิภาพสูง** - ออกแบบมาเพื่อความเร็วและประสิทธิภาพ
- 🔒 **ความปลอดภัย** - มีระบบรักษาความปลอดภัยในตัว
- 📦 **Modular** - ระบบ module ที่ยืดหยุ่นและขยายได้
- 🎨 **MVC Pattern** - รองรับ Model-View-Controller
- 🌐 **Multi-language** - รองรับหลายภาษา
- 📱 **Responsive** - รองรับทุกอุปกรณ์

## ตัวอย่างโค้ด

```php
<?php
namespace App\Controllers;

use Kotchasan\Controller;
use Kotchasan\Http\Request;

class HomeController extends Controller
{
    public function index(Request $request)
    {
        $data = [
            'title' => 'ยินดีต้อนรับสู่ Kotchasan',
            'message' => 'Framework ที่มีประสิทธิภาพสูง'
        ];

        return $this->view('home', $data);
    }
}
```

## การติดตั้ง

### ข้อกำหนดระบบ

- PHP 8.0 หรือสูงกว่า
- Composer
- Web Server (Apache/Nginx)
- MySQL 5.7 หรือสูงกว่า

### ขั้นตอนการติดตั้ง

1. **Clone Repository**
   ```bash
   git clone https://github.com/your-org/kotchasan.git
   cd kotchasan
   ```

2. **ติดตั้ง Dependencies**
   ```bash
   composer install
   ```

3. **ตั้งค่า Environment**
   ```bash
   cp .env.example .env
   # แก้ไขไฟล์ .env ตามการตั้งค่าของคุณ
   ```

4. **ตั้งค่าฐานข้อมูล**
   ```bash
   php artisan migrate
   php artisan db:seed
   ```

5. **เริ่มต้น Development Server**
   ```bash
   php artisan serve
   ```

## การใช้งานเบื้องต้น

### สร้าง Controller ใหม่

```php
<?php
namespace App\Controllers;

use Kotchasan\Controller;

class ApiController extends Controller
{
    public function users()
    {
        $users = User::all();
        return $this->json($users);
    }
}
```

### สร้าง Model

```php
<?php
namespace App\Models;

use Kotchasan\Model;

class User extends Model
{
    protected $table = 'users';

    protected $fillable = [
        'name', 'email', 'password'
    ];

    protected $hidden = [
        'password'
    ];
}
```

### สร้าง View

```html
<!DOCTYPE html>
<html lang="th">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $title ?></title>
</head>
<body>
    <h1><?= $title ?></h1>
    <p><?= $message ?></p>
</body>
</html>
```

## การตั้งค่า Routing

```php
<?php
// routes/web.php

use App\Controllers\HomeController;
use App\Controllers\ApiController;

// Web Routes
Route::get('/', [HomeController::class, 'index']);
Route::get('/about', [HomeController::class, 'about']);

// API Routes
Route::prefix('api')->group(function () {
    Route::get('/users', [ApiController::class, 'users']);
    Route::post('/users', [ApiController::class, 'store']);
});
```

## ระบบความปลอดภัย

### การป้องกัน CSRF

```php
// ในฟอร์ม
<form method="POST">
    <?= csrf_token() ?>
    <!-- ฟิลด์อื่นๆ -->
</form>

// ตรวจสอบใน Controller
public function store(Request $request)
{
    if (!$request->validateCsrf()) {
        return $this->error('Invalid CSRF token');
    }
    // ประมวลผลข้อมูล
}
```

### การเข้ารหัสรหัสผ่าน

```php
// เข้ารหัส
$hashedPassword = password_hash($password, PASSWORD_DEFAULT);

// ตรวจสอบ
if (password_verify($password, $hashedPassword)) {
    // รหัสผ่านถูกต้อง
}
```

## การ Debug และ Logging

```php
use Kotchasan\Log\Logger;

// Log ข้อความ
Logger::info('User logged in', ['user_id' => $userId]);
Logger::error('Database connection failed', ['error' => $error]);

// Debug mode
if (config('app.debug')) {
    dump($variable);
    dd($data); // dump and die
}
```

## การทดสอบ

### Unit Testing

```php
<?php
namespace Tests\Unit;

use PHPUnit\Framework\TestCase;
use App\Models\User;

class UserTest extends TestCase
{
    public function testUserCreation()
    {
        $user = new User([
            'name' => 'John Doe',
            'email' => 'john@example.com'
        ]);

        $this->assertEquals('John Doe', $user->name);
        $this->assertEquals('john@example.com', $user->email);
    }
}
```

## การ Deploy

### Production Setup

1. **ตั้งค่า Environment**
   ```bash
   # ตั้งค่า APP_ENV=production
   # ตั้งค่า APP_DEBUG=false
   ```

2. **Optimize Application**
   ```bash
   php artisan optimize
   php artisan config:cache
   php artisan route:cache
   ```

3. **ตั้งค่า Web Server**
   ```nginx
   server {
       listen 80;
       server_name your-domain.com;
       root /path/to/kotchasan/public;

       location / {
           try_files $uri $uri/ /index.php?$query_string;
       }

       location ~ \.php$ {
           fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
           fastcgi_index index.php;
           include fastcgi_params;
       }
   }
   ```

## ทรัพยากรเพิ่มเติม

- 📚 [คู่มือการใช้งาน](getting-started.md)
- 🔧 [การติดตั้ง](installation.md)
- 📖 [API Reference](api-reference.md)
- 🐛 [การแก้ไขปัญหา](troubleshooting.md)
- 💬 [Community Forum](https://forum.kotchasan.com)

## การสนับสนุน

หากคุณพบปัญหาหรือต้องการความช่วยเหลือ:

- 📧 Email: support@kotchasan.com
- 💬 Discord: [Kotchasan Community](https://discord.gg/kotchasan)
- 🐛 GitHub Issues: [Report Bug](https://github.com/kotchasan/framework/issues)

---

**Kotchasan Framework** - พัฒนาเว็บแอปพลิเคชันได้อย่างรวดเร็วและปลอดภัย
