# การติดตั้ง Kotchasan Framework

## ข้อกำหนดระบบ

### ข้อกำหนดขั้นต่ำ

- **PHP**: 8.0 หรือสูงกว่า
- **Composer**: 2.0 หรือสูงกว่า
- **Web Server**: Apache 2.4+ หรือ Nginx 1.18+
- **Database**: MySQL 5.7+ หรือ MariaDB 10.3+
- **Memory**: 512MB RAM (แนะนำ 1GB+)
- **Storage**: 100MB พื้นที่ว่าง

### ข้อกำหนดที่แนะนำ

- **PHP**: 8.1 หรือ 8.2
- **Memory**: 2GB RAM
- **Storage**: 1GB พื้นที่ว่าง
- **SSL Certificate**: สำหรับ Production

## การติดตั้ง

### 1. ติดตั้งผ่าน Composer (แนะนำ)

```bash
# สร้างโปรเจ็คใหม่
composer create-project kotchasan/framework my-project

# เข้าไปในโฟลเดอร์โปรเจ็ค
cd my-project
```

### 2. ติดตั้งผ่าน Git Clone

```bash
# Clone repository
git clone https://github.com/kotchasan/framework.git my-project
cd my-project

# ติดตั้ง dependencies
composer install
```

### 3. ติดตั้งผ่าน Download

```bash
# ดาวน์โหลดไฟล์ ZIP
wget https://github.com/kotchasan/framework/archive/main.zip
unzip main.zip
mv framework-main my-project
cd my-project

# ติดตั้ง dependencies
composer install
```

## การตั้งค่า Environment

### 1. คัดลอกไฟล์ Environment

```bash
cp .env.example .env
```

### 2. แก้ไขไฟล์ .env

```env
# Application
APP_NAME="Kotchasan Framework"
APP_ENV=local
APP_DEBUG=true
APP_URL=http://localhost

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=kotchasan
DB_USERNAME=root
DB_PASSWORD=

# Cache
CACHE_DRIVER=file
SESSION_DRIVER=file

# Mail
MAIL_MAILER=smtp
MAIL_HOST=localhost
MAIL_PORT=587
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
```

### 3. สร้าง Application Key

```bash
php artisan key:generate
```

## การตั้งค่าฐานข้อมูล

### 1. สร้างฐานข้อมูล

```sql
-- MySQL
CREATE DATABASE kotchasan CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- หรือใช้ MariaDB
CREATE DATABASE kotchasan CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

### 2. รัน Migration

```bash
# รัน migration ทั้งหมด
php artisan migrate

# หรือรันทีละไฟล์
php artisan migrate --step
```

### 3. Seed ข้อมูลเริ่มต้น

```bash
# Seed ข้อมูลเริ่มต้น
php artisan db:seed

# หรือ seed เฉพาะ seeder ที่ต้องการ
php artisan db:seed --class=UserSeeder
```

## การตั้งค่า Web Server

### Apache Configuration

สร้างไฟล์ `.htaccess` ในโฟลเดอร์ `public`:

```apache
<IfModule mod_rewrite.c>
    RewriteEngine On

    # Handle Angular and other client-side routes
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^(.*)$ index.php [QSA,L]
</IfModule>

# Security headers
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
</IfModule>
```

### Nginx Configuration

```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /path/to/kotchasan/public;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }
}
```

## การตั้งค่า SSL (HTTPS)

### 1. ใช้ Let's Encrypt (แนะนำ)

```bash
# ติดตั้ง Certbot
sudo apt install certbot python3-certbot-nginx

# สร้าง SSL Certificate
sudo certbot --nginx -d your-domain.com
```

### 2. ใช้ Self-Signed Certificate

```bash
# สร้าง private key
openssl genrsa -out server.key 2048

# สร้าง certificate
openssl req -new -x509 -key server.key -out server.crt -days 365
```

## การตั้งค่า Cache

### 1. File Cache (Default)

```env
CACHE_DRIVER=file
```

### 2. Redis Cache

```env
CACHE_DRIVER=redis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
```

### 3. Memcached

```env
CACHE_DRIVER=memcached
MEMCACHED_HOST=127.0.0.1
MEMCACHED_PORT=11211
```

## การตั้งค่า Queue

### 1. Database Queue

```env
QUEUE_CONNECTION=database
```

```bash
# สร้างตาราง queue
php artisan queue:table
php artisan migrate
```

### 2. Redis Queue

```env
QUEUE_CONNECTION=redis
```

## การตั้งค่า Mail

### 1. SMTP Configuration

```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_ENCRYPTION=tls
```

### 2. Mailgun

```env
MAIL_MAILER=mailgun
MAILGUN_DOMAIN=your-domain.mailgun.org
MAILGUN_SECRET=your-mailgun-secret
```

## การตั้งค่า Logging

### 1. File Logging

```env
LOG_CHANNEL=stack
LOG_LEVEL=debug
```

### 2. Syslog

```env
LOG_CHANNEL=syslog
```

### 3. Custom Log Channel

```php
// config/logging.php
'channels' => [
    'custom' => [
        'driver' => 'single',
        'path' => storage_path('logs/custom.log'),
        'level' => 'debug',
    ],
],
```

## การทดสอบการติดตั้ง

### 1. ตรวจสอบ PHP Version

```bash
php -v
# ควรแสดง PHP 8.0 หรือสูงกว่า
```

### 2. ตรวจสอบ Composer

```bash
composer --version
# ควรแสดง Composer 2.0 หรือสูงกว่า
```

### 3. ตรวจสอบ Extension ที่จำเป็น

```bash
php -m | grep -E "(pdo|mbstring|openssl|curl|json|zip)"
```

### 4. ทดสอบ Application

```bash
# เริ่ม development server
php artisan serve

# เปิดเบราว์เซอร์ไปที่ http://localhost:8000
```

## การแก้ไขปัญหา

### ปัญหาที่พบบ่อย

#### 1. Permission Issues

```bash
# ตั้งค่า permission
sudo chown -R www-data:www-data storage bootstrap/cache
sudo chmod -R 775 storage bootstrap/cache
```

#### 2. Composer Memory Limit

```bash
# เพิ่ม memory limit
php -d memory_limit=2G /usr/local/bin/composer install
```

#### 3. Database Connection Error

```bash
# ตรวจสอบการเชื่อมต่อฐานข้อมูล
php artisan tinker
>>> DB::connection()->getPdo();
```

#### 4. Cache Issues

```bash
# ล้าง cache
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

## การอัปเดต

### 1. อัปเดต Dependencies

```bash
composer update
```

### 2. อัปเดต Database Schema

```bash
php artisan migrate
```

### 3. อัปเดต Cache

```bash
php artisan optimize
```

## การสำรองข้อมูล

### 1. สำรองฐานข้อมูล

```bash
# MySQL
mysqldump -u root -p kotchasan > backup.sql

# MariaDB
mysqldump -u root -p kotchasan > backup.sql
```

### 2. สำรองไฟล์

```bash
# สำรองไฟล์สำคัญ
tar -czf backup.tar.gz storage/ .env config/
```

## การติดตั้งบน Production

### 1. ตั้งค่า Environment

```env
APP_ENV=production
APP_DEBUG=false
APP_URL=https://your-domain.com
```

### 2. Optimize Application

```bash
php artisan optimize
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### 3. ตั้งค่า Process Manager

```bash
# ติดตั้ง Supervisor
sudo apt install supervisor

# สร้างไฟล์ configuration
sudo nano /etc/supervisor/conf.d/kotchasan.conf
```

```ini
[program:kotchasan]
command=php /path/to/kotchasan/artisan queue:work
directory=/path/to/kotchasan
autostart=true
autorestart=true
user=www-data
redirect_stderr=true
stdout_logfile=/path/to/kotchasan/storage/logs/worker.log
```

---

**หมายเหตุ**: เอกสารนี้ครอบคลุมการติดตั้งพื้นฐาน สำหรับข้อมูลเพิ่มเติมโปรดดูที่ [Getting Started Guide](getting-started.md)

