# Documentation System Usage Guide

## Overview

This documentation system is a real-time Markdown file display system designed to showcase Kotchasan Framework documentation.

## Key Features

### 🌐 Multi-language Support
- Supports Thai and English
- Real-time language switching
- Separate documentation files per language

### 🎨 Themes
- Light Mode
- Dark Mode
- Instant theme switching

### 📱 Responsive Design
- Supports all screen sizes
- Mobile-first design
- Touch-friendly interface

## Basic Usage

### 1. Opening the Documentation System

```bash
# Open index.html in browser
open index.html

# Or use local server
python3 -m http.server 8000
# Open http://localhost:8000
```

### 2. Navigation

#### Left Sidebar Menu
- Shows all available documents
- Click to switch documents
- Shows selected document status

#### Language Switching
- Click 🌐 TH/EN button
- System loads selected language documentation
- Supports real-time language switching

#### Theme Switching
- Click 🌙 Dark/☀️ Light button
- Changes background and text colors
- Saves settings in localStorage

### 3. Reading Documentation

#### Markdown Support
- Full Markdown syntax support
- Code highlighting for PHP, JavaScript, CSS
- Tables, lists, links

#### Code Blocks
```markdown
```php
<?php
echo "Hello World";
```
```

#### Tables
```markdown
| Column 1 | Column 2 |
|----------|----------|
| Data 1   | Data 2   |
```

## Document Management

### File Structure

```
docs/
├── th/                    # Thai documentation
│   ├── index.md          # Home page
│   ├── installation.md   # Installation guide
│   ├── getting-started.md # Getting started guide
│   ├── api-reference.md   # API Reference
│   └── usage-guide.md     # Usage guide
└── en/                    # English documentation
    ├── index.md
    ├── installation.md
    ├── getting-started.md
    ├── api-reference.md
    └── usage-guide.md
```

### Adding New Documents

#### 1. Create Markdown Files

```bash
# Create new files
touch docs/th/new-document.md
touch docs/en/new-document.md
```

#### 2. Write Content

```markdown
# Document Title

## Introduction
Document content

## Main Content
- Item 1
- Item 2

## Summary
Content summary
```

#### 3. System Auto-loads

Files appear in left sidebar immediately

### File Naming

#### Naming Rules
- Use English only
- Use `-` instead of spaces
- Use lowercase only
- Avoid special characters

#### File Name Examples
```
✅ Correct:
- getting-started.md
- api-reference.md
- troubleshooting.md

❌ Wrong:
- Getting Started.md
- api_reference.md
- การใช้งาน.md
```

## Customization

### Changing Colors

#### Edit CSS Variables

```css
:root {
  --bg-primary: #ffffff;        /* Main background */
  --bg-secondary: #f8f9fa;     /* Secondary background */
  --text-primary: #212529;      /* Main text color */
  --accent: #0d6efd;           /* Accent color */
}
```

#### Dark Mode
```css
[data-theme="dark"] {
  --bg-primary: #1a1d23;
  --bg-secondary: #22252b;
  --text-primary: #e9ecef;
  --accent: #4dabf7;
}
```

### Adding Features

#### Add New Button

```html
<button class="btn" id="newFeatureBtn">🆕 New Feature</button>
```

```javascript
document.getElementById('newFeatureBtn').addEventListener('click', () => {
    // New functionality
});
```

#### Add Menu

```javascript
// Add new menu
function addCustomMenu() {
    const menu = document.getElementById('fileTree');
    const newItem = document.createElement('div');
    newItem.className = 'file-item';
    newItem.textContent = 'New Menu';
    newItem.onclick = () => loadCustomContent();
    menu.appendChild(newItem);
}
```

## Troubleshooting

### Common Issues

#### 1. Files Not Loading

**Causes:**
- Wrong file name
- File not in correct location
- Incorrect file permissions

**Solutions:**
```bash
# Check file names
ls -la docs/th/

# Check permissions
chmod 644 docs/th/*.md

# Check structure
tree docs/
```

#### 2. Theme Not Changing

**Causes:**
- JavaScript error
- CSS not loading
- Browser cache

**Solutions:**
```javascript
// Check console
console.log('Theme:', currentTheme);

// Clear cache
localStorage.clear();
location.reload();
```

#### 3. Language Not Switching

**Causes:**
- .md file not available in selected language
- File names don't match
- JavaScript error

**Solutions:**
```javascript
// Check available files
console.log('Available docs:', Object.keys(docs));

// Check language
console.log('Current lang:', currentLang);
```

### Debugging

#### Open Developer Tools

```javascript
// Check status
console.log('Current language:', currentLang);
console.log('Current theme:', currentTheme);
console.log('Available documents:', Object.keys(docs));
console.log('Current document:', currentDoc);
```

#### Check Network

1. Open Developer Tools (F12)
2. Go to Network tab
3. Watch .md file loading
4. Check HTTP status codes

## Deployment

### 1. Prepare Files

```bash
# Create production files
cp -r docs/ production/
cp index.html production/
cp README.md production/
```

### 2. Upload

```bash
# Use rsync
rsync -avz production/ user@server:/var/www/docs/

# Use scp
scp -r production/* user@server:/var/www/docs/
```

### 3. Web Server Configuration

#### Apache
```apache
<VirtualHost *:80>
    ServerName docs.yourdomain.com
    DocumentRoot /var/www/docs

    <Directory "/var/www/docs">
        AllowOverride All
        Require all granted
    </Directory>
</VirtualHost>
```

#### Nginx
```nginx
server {
    listen 80;
    server_name docs.yourdomain.com;
    root /var/www/docs;
    index index.html;

    location / {
        try_files $uri $uri/ /index.html;
    }

    location ~ \.md$ {
        add_header Content-Type text/plain;
    }
}
```

## Maintenance

### Updating Documentation

```bash
# Update documentation
git pull origin main

# Check changes
git diff HEAD~1

# Test functionality
python3 -m http.server 8000
```

### Backup

```bash
# Backup files
tar -czf docs-backup-$(date +%Y%m%d).tar.gz docs/

# Backup database (if any)
mysqldump -u user -p database > backup.sql
```

### Usage Tracking

```javascript
// Add Google Analytics
gtag('config', 'GA_MEASUREMENT_ID');

// Track language changes
function trackLanguageChange(lang) {
    gtag('event', 'language_change', {
        'language': lang
    });
}

// Track theme changes
function trackThemeChange(theme) {
    gtag('event', 'theme_change', {
        'theme': theme
    });
}
```

## Further Development

### Adding New Features

#### 1. Search Functionality

```javascript
// Add search function
function addSearch() {
    const searchInput = document.createElement('input');
    searchInput.type = 'text';
    searchInput.placeholder = 'Search documents...';
    searchInput.onkeyup = (e) => searchDocuments(e.target.value);
    document.querySelector('.header-controls').appendChild(searchInput);
}

function searchDocuments(query) {
    const results = Object.keys(docs).filter(doc =>
        doc.toLowerCase().includes(query.toLowerCase())
    );
    // Display search results
}
```

#### 2. Print Functionality

```javascript
// Add print function
function addPrintButton() {
    const printBtn = document.createElement('button');
    printBtn.className = 'btn';
    printBtn.innerHTML = '🖨️ Print';
    printBtn.onclick = () => window.print();
    document.querySelector('.header-controls').appendChild(printBtn);
}
```

#### 3. Export Functionality

```javascript
// Add export function
function addExportButton() {
    const exportBtn = document.createElement('button');
    exportBtn.className = 'btn';
    exportBtn.innerHTML = '📄 Export';
    exportBtn.onclick = () => exportDocument();
    document.querySelector('.header-controls').appendChild(exportBtn);
}

function exportDocument() {
    const content = document.getElementById('content').innerHTML;
    const blob = new Blob([content], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${currentDoc}.html`;
    a.click();
}
```

---

**Note**: This guide covers the documentation system usage. For more information, please refer to the [API Reference](api-reference.md)

