// Main Application Logic

// Modal Functions
function openModal(modalId) {
  const modal = document.getElementById(modalId);
  if (modal) {
    modal.classList.add('active');
  }
}

function closeModal(modalId) {
  const modal = document.getElementById(modalId);
  if (modal) {
    modal.classList.remove('active');
  }
}

// Navigation
function navigateToPage(pageName) {
  // Hide all pages
  document.querySelectorAll('.page').forEach(page => {
    page.classList.remove('active');
  });

  // Show selected page
  const targetPage = document.getElementById(`${pageName}-page`);
  if (targetPage) {
    targetPage.classList.add('active');
  }

  // Update sidebar active state
  document.querySelectorAll('.sidebar-nav li').forEach(li => {
    li.classList.remove('active');
  });

  const activeLink = document.querySelector(`[data-page="${pageName}"]`);
  if (activeLink) {
    activeLink.parentElement.classList.add('active');
  }

  // Load page-specific data
  switch (pageName) {
    case 'dashboard':
      initDashboard();
      break;
    case 'users':
      initUserManagement();
      break;
    case 'products':
      loadProducts();
      break;
    case 'orders':
      loadOrders();
      break;
    case 'analytics':
      initCharts();
      break;
  }
}

// Initialize Application
document.addEventListener('DOMContentLoaded', () => {
  // Skip initialization on login page
  if (window.location.pathname.includes('login.html')) {
    return;
  }

  // Sidebar toggle
  const sidebarToggle = document.getElementById('sidebarToggle');
  const sidebar = document.getElementById('sidebar');

  if (sidebarToggle && sidebar) {
    sidebarToggle.addEventListener('click', () => {
      sidebar.classList.toggle('collapsed');
    });
  }

  // Mobile toggle
  const mobileToggle = document.getElementById('mobileToggle');
  if (mobileToggle && sidebar) {
    mobileToggle.addEventListener('click', () => {
      sidebar.classList.toggle('active');
    });
  }

  // Theme toggle
  const themeToggle = document.getElementById('themeToggle');
  if (themeToggle) {
    themeToggle.addEventListener('click', () => {
      const currentTheme = document.documentElement.getAttribute('data-theme');
      const newTheme = currentTheme === 'dark' ? 'light' : 'dark';

      document.documentElement.setAttribute('data-theme', newTheme);
      storage.set('theme', newTheme);

      const icon = themeToggle.querySelector('i');
      if (newTheme === 'dark') {
        icon.classList.remove('fa-moon');
        icon.classList.add('fa-sun');
      } else {
        icon.classList.remove('fa-sun');
        icon.classList.add('fa-moon');
      }
    });

    // Load saved theme
    const savedTheme = storage.get('theme');
    if (savedTheme) {
      document.documentElement.setAttribute('data-theme', savedTheme);
      const icon = themeToggle.querySelector('i');
      if (savedTheme === 'dark') {
        icon.classList.remove('fa-moon');
        icon.classList.add('fa-sun');
      }
    }
  }

  // Notification button
  const notificationBtn = document.getElementById('notificationBtn');
  if (notificationBtn) {
    notificationBtn.addEventListener('click', () => {
      showToast('You have 3 new notifications', 'info', 'Notifications');
    });
  }

  // User dropdown
  const userMenu = document.querySelector('.user-menu img');
  const userDropdown = document.getElementById('userDropdown');
  const userMenuContainer = document.querySelector('.user-menu');

  if (userMenu && userDropdown) {
    userMenu.addEventListener('click', (e) => {
      e.stopPropagation();
      userDropdown.classList.toggle('active');
    });

    // Prevent dropdown from closing when clicking inside it
    if (userMenuContainer) {
      userMenuContainer.addEventListener('click', (e) => {
        e.stopPropagation();
      });
    }

    // Close dropdown when clicking outside
    document.addEventListener('click', (e) => {
      if (!userMenuContainer.contains(e.target)) {
        userDropdown.classList.remove('active');
      }
    });
  }

  // Navigation links
  document.querySelectorAll('[data-page]').forEach(link => {
    link.addEventListener('click', (e) => {
      e.preventDefault();
      const pageName = link.getAttribute('data-page');
      navigateToPage(pageName);
    });
  });

  // Modal close buttons
  document.querySelectorAll('[data-modal]').forEach(btn => {
    btn.addEventListener('click', () => {
      const modalId = btn.getAttribute('data-modal');
      closeModal(modalId);
    });
  });

  // Close modal on outside click
  document.querySelectorAll('.modal').forEach(modal => {
    modal.addEventListener('click', (e) => {
      if (e.target === modal) {
        modal.classList.remove('active');
      }
    });
  });

  // Add product button
  const addProductBtn = document.getElementById('addProductBtn');
  if (addProductBtn) {
    addProductBtn.addEventListener('click', () => {
      // Reset form to add mode
      const productForm = document.getElementById('productForm');
      const modalTitle = document.querySelector('#productModal .modal-header h2');

      if (productForm) {
        productForm.reset();
        delete productForm.dataset.editProductId;
        delete productForm.dataset.editMode;
      }

      if (modalTitle) {
        modalTitle.textContent = 'Add Product';
      }

      openModal('productModal');
    });
  }

  // Product form
  const productForm = document.getElementById('productForm');
  if (productForm) {
    productForm.addEventListener('submit', async (e) => {
      e.preventDefault();

      if (!validateForm(productForm)) {
        showToast('Please fill in all required fields', 'error');
        return;
      }

      const formData = new FormData(productForm);
      const isEditMode = productForm.dataset.editMode === 'true';
      const productId = productForm.dataset.editProductId;

      const productData = {
        name: formData.get('name'),
        category: formData.get('category'),
        price: formData.get('price'),
        stock: formData.get('stock'),
        description: formData.get('description')
      };

      try {
        showLoading();

        let response;
        if (isEditMode && productId) {
          response = await mockApi.updateProduct(productId, productData);
          if (response.success) {
            showToast('Product updated successfully', 'success');
          }
        } else {
          response = await mockApi.createProduct(productData);
          if (response.success) {
            showToast('Product added successfully', 'success');
          }
        }

        if (response.success) {
          closeModal('productModal');
          loadProducts();
          productForm.reset();
          delete productForm.dataset.editProductId;
          delete productForm.dataset.editMode;

          // Reset modal title
          const modalTitle = document.querySelector('#productModal .modal-header h2');
          if (modalTitle) {
            modalTitle.textContent = 'Add Product';
          }
        }
      } catch (error) {
        showToast(isEditMode ? 'Failed to update product' : 'Failed to add product', 'error');
      } finally {
        hideLoading();
      }
    });
  }

  // Order form
  const orderForm = document.getElementById('orderForm');
  if (orderForm) {
    orderForm.addEventListener('submit', async (e) => {
      e.preventDefault();

      if (!validateForm(orderForm)) {
        showToast('Please fill in all required fields', 'error');
        return;
      }

      const formData = new FormData(orderForm);
      const orderId = orderForm.dataset.editOrderId;

      const orderData = {
        customer: formData.get('customer'),
        products: parseInt(formData.get('products')),
        total: parseFloat(formData.get('total')),
        status: formData.get('status')
      };

      try {
        showLoading();

        const response = await mockApi.updateOrder(orderId, orderData);

        if (response.success) {
          showToast('Order updated successfully', 'success');
          closeModal('orderModal');
          loadOrders();
          orderForm.reset();
          delete orderForm.dataset.editOrderId;
          delete orderForm.dataset.editMode;
        }
      } catch (error) {
        showToast('Failed to update order', 'error');
      } finally {
        hideLoading();
      }
    });
  }

  // Settings form
  const profileForm = document.getElementById('profileForm');
  if (profileForm) {
    profileForm.addEventListener('submit', (e) => {
      e.preventDefault();
      showToast('Profile updated successfully', 'success');
    });
  }

  // Theme select in settings
  const themeSelect = document.getElementById('themeSelect');
  if (themeSelect) {
    themeSelect.value = storage.get('theme') || 'light';

    themeSelect.addEventListener('change', (e) => {
      const theme = e.target.value;
      document.documentElement.setAttribute('data-theme', theme);
      storage.set('theme', theme);
      showToast('Theme updated', 'success');
    });
  }

  // Initialize dashboard on load
  initDashboard();

  // Close sidebar on mobile when clicking outside
  document.addEventListener('click', (e) => {
    if (window.innerWidth <= 768) {
      if (!sidebar.contains(e.target) && !mobileToggle.contains(e.target)) {
        sidebar.classList.remove('active');
      }
    }
  });
});