// API Configuration
const API_BASE_URL = ''; // Replace with your API URL

// API Helper Functions
const api = {
  // Generic request handler
  async request(endpoint, options = {}) {
    const token = storage.get('authToken');

    const config = {
      headers: {
        'Content-Type': 'application/json',
        ...(token && {'Authorization': `Bearer ${token}`}),
        ...options.headers
      },
      ...options
    };

    try {
      const response = await fetch(`${API_BASE_URL}${endpoint}`, config);

      if (!response.ok) {
        if (response.status === 401) {
          // Unauthorized - redirect to login
          window.location.href = 'login.html';
          throw new Error('Unauthorized');
        }
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('API Error:', error);
      throw error;
    }
  },

  // GET request
  async get(endpoint) {
    return this.request(endpoint, {method: 'GET'});
  },

  // POST request
  async post(endpoint, data) {
    return this.request(endpoint, {
      method: 'POST',
      body: JSON.stringify(data)
    });
  },

  // PUT request
  async put(endpoint, data) {
    return this.request(endpoint, {
      method: 'PUT',
      body: JSON.stringify(data)
    });
  },

  // DELETE request
  async delete(endpoint) {
    return this.request(endpoint, {method: 'DELETE'});
  }
};

// Mock Data Generator (for demo purposes)
const mockData = {
  // Generate mock users
  generateUsers(count = 20) {
    const roles = ['admin', 'user', 'moderator'];
    const statuses = ['active', 'inactive'];
    const users = [];

    for (let i = 1; i <= count; i++) {
      users.push({
        id: i,
        name: `User ${i}`,
        email: `user${i}@example.com`,
        role: roles[Math.floor(Math.random() * roles.length)],
        status: statuses[Math.floor(Math.random() * statuses.length)],
        avatar: `https://i.pravatar.cc/150?img=${i}`,
        joinedDate: new Date(Date.now() - Math.random() * 365 * 24 * 60 * 60 * 1000).toISOString()
      });
    }

    return users;
  },

  // Generate mock products
  generateProducts(count = 12) {
    const categories = ['electronics', 'clothing', 'food', 'books'];
    const products = [];

    for (let i = 1; i <= count; i++) {
      products.push({
        id: i,
        name: `Product ${i}`,
        category: categories[Math.floor(Math.random() * categories.length)],
        price: (Math.random() * 1000).toFixed(2),
        stock: Math.floor(Math.random() * 100),
        image: `https://picsum.photos/400/300?random=${i}`,
        description: `Description for product ${i}`
      });
    }

    return products;
  },

  // Generate mock orders
  generateOrders(count = 15) {
    const statuses = ['pending', 'processing', 'completed', 'cancelled'];
    const orders = [];

    for (let i = 1; i <= count; i++) {
      orders.push({
        id: `ORD-${1000 + i}`,
        customer: `Customer ${i}`,
        products: Math.floor(Math.random() * 5) + 1,
        total: (Math.random() * 1000).toFixed(2),
        status: statuses[Math.floor(Math.random() * statuses.length)],
        date: new Date(Date.now() - Math.random() * 30 * 24 * 60 * 60 * 1000).toISOString()
      });
    }

    return orders;
  },

  // Generate mock activities
  generateActivities(count = 10) {
    const types = ['user', 'order', 'product', 'system'];
    const actions = ['created', 'updated', 'deleted', 'logged in'];
    const activities = [];

    for (let i = 1; i <= count; i++) {
      const type = types[Math.floor(Math.random() * types.length)];
      activities.push({
        id: i,
        type: type,
        title: `${type.charAt(0).toUpperCase() + type.slice(1)} ${actions[Math.floor(Math.random() * actions.length)]}`,
        description: `A ${type} was ${actions[Math.floor(Math.random() * actions.length)]}`,
        time: new Date(Date.now() - Math.random() * 24 * 60 * 60 * 1000).toISOString()
      });
    }

    return activities;
  },

  // Generate chart data
  generateChartData(points = 12) {
    const labels = [];
    const data = [];

    for (let i = 0; i < points; i++) {
      labels.push(`Month ${i + 1}`);
      data.push(Math.floor(Math.random() * 10000));
    }

    return {labels, data};
  }
};

// Mock API calls (replace with real API calls)
const mockApi = {
  async login(email, password) {
    console.log('Mock login with', email, password);
    // Simulate API delay
    await new Promise(resolve => setTimeout(resolve, 1000));

    if (email === 'admin@example.com' && password === 'admin123') {
      return {
        success: true,
        token: 'mock-jwt-token-' + Date.now(),
        user: {
          id: 1,
          name: 'Admin User',
          email: email,
          role: 'admin'
        }
      };
    }

    throw new Error('Invalid credentials');
  },

  async getDashboardStats() {
    await new Promise(resolve => setTimeout(resolve, 500));

    return {
      totalUsers: 1234,
      totalRevenue: 45678.90,
      totalOrders: 567,
      totalProducts: 89
    };
  },

  async getUsers(page = 1, limit = 10, search = '', role = '') {
    await new Promise(resolve => setTimeout(resolve, 500));

    let users = mockData.generateUsers(50);

    if (search) {
      users = users.filter(u =>
        u.name.toLowerCase().includes(search.toLowerCase()) ||
        u.email.toLowerCase().includes(search.toLowerCase())
      );
    }

    if (role) {
      users = users.filter(u => u.role === role);
    }

    const start = (page - 1) * limit;
    const end = start + limit;

    return {
      users: users.slice(start, end),
      total: users.length,
      page,
      totalPages: Math.ceil(users.length / limit)
    };
  },

  async getProducts() {
    await new Promise(resolve => setTimeout(resolve, 500));
    return mockData.generateProducts();
  },

  async getOrders() {
    await new Promise(resolve => setTimeout(resolve, 500));
    return mockData.generateOrders();
  },

  async getActivities() {
    await new Promise(resolve => setTimeout(resolve, 500));
    return mockData.generateActivities();
  },

  async createUser(userData) {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {
      success: true,
      user: {
        id: Date.now(),
        ...userData,
        avatar: `https://i.pravatar.cc/150?img=${Math.floor(Math.random() * 70)}`,
        joinedDate: new Date().toISOString()
      }
    };
  },

  async updateUser(id, userData) {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {
      success: true,
      user: {
        id,
        ...userData,
        avatar: `https://i.pravatar.cc/150?img=${Math.floor(Math.random() * 70)}`,
        joinedDate: new Date().toISOString()
      }
    };
  },

  async deleteUser(id) {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {success: true};
  },

  async createProduct(productData) {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {
      success: true,
      product: {
        id: Date.now(),
        ...productData,
        image: `https://picsum.photos/400/300?random=${Date.now()}`
      }
    };
  },

  async updateProduct(id, productData) {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {
      success: true,
      product: {
        id,
        ...productData,
        image: `https://picsum.photos/400/300?random=${id}`
      }
    };
  },

  async deleteProduct(id) {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {success: true};
  },

  async updateOrder(id, orderData) {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {
      success: true,
      order: {
        id,
        ...orderData,
        date: new Date().toISOString()
      }
    };
  },

  async deleteOrder(id) {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {success: true};
  },

  async getOrderDetails(id) {
    await new Promise(resolve => setTimeout(resolve, 500));

    // Generate detailed order information
    const subtotal = Math.random() * 500 + 100;
    const shippingCost = 15.99;
    const tax = subtotal * 0.08;
    const total = subtotal + shippingCost + tax;

    const products = ['Laptop', 'Mouse', 'Keyboard', 'Monitor', 'Headphones', 'Webcam'];
    const numItems = Math.floor(Math.random() * 3) + 1;
    const items = [];

    for (let i = 0; i < numItems; i++) {
      const product = products[Math.floor(Math.random() * products.length)];
      const quantity = Math.floor(Math.random() * 3) + 1;
      const price = Math.random() * 200 + 50;
      items.push({product, quantity, price});
    }

    return {
      id: id,
      status: ['pending', 'processing', 'completed'][Math.floor(Math.random() * 3)],
      date: new Date(Date.now() - Math.random() * 30 * 24 * 60 * 60 * 1000).toISOString(),
      customer: {
        name: 'John Doe',
        email: 'john.doe@example.com',
        phone: '+1 (555) 123-4567',
        address: '123 Main Street, New York, NY 10001, USA'
      },
      items: items,
      payment: {
        method: 'Credit Card (****1234)',
        status: 'Paid',
        transactionId: 'TXN-' + Math.random().toString(36).substr(2, 9).toUpperCase()
      },
      shipping: {
        method: 'Standard Shipping',
        trackingNumber: 'TRK-' + Math.random().toString(36).substr(2, 12).toUpperCase(),
        estimatedDelivery: new Date(Date.now() + Math.random() * 7 * 24 * 60 * 60 * 1000).toISOString()
      },
      subtotal: subtotal,
      shippingCost: shippingCost,
      tax: tax,
      total: total
    };
  }
};