// Chart.js Configuration and Functions

let revenueChart, userChart, salesChart, categoryChart, trafficChart;

// Initialize all charts
function initCharts() {
  initRevenueChart();
  initUserChart();
  initSalesChart();
  initCategoryChart();
  initTrafficChart();
}

// Revenue Chart
function initRevenueChart() {
  const ctx = document.getElementById('revenueChart');
  if (!ctx) return;

  // Destroy existing chart if it exists
  if (revenueChart) {
    revenueChart.destroy();
  }

  const data = mockData.generateChartData(12);

  revenueChart = new Chart(ctx, {
    type: 'line',
    data: {
      labels: data.labels,
      datasets: [{
        label: 'Revenue',
        data: data.data,
        borderColor: '#6366f1',
        backgroundColor: 'rgba(99, 102, 241, 0.1)',
        tension: 0.4,
        fill: true
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          display: false
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              return 'Revenue: ' + formatCurrency(context.parsed.y);
            }
          }
        }
      },
      scales: {
        y: {
          beginAtZero: true,
          ticks: {
            callback: function(value) {
              return '$' + value.toLocaleString();
            }
          }
        }
      }
    }
  });
}

// User Growth Chart
function initUserChart() {
  const ctx = document.getElementById('userChart');
  if (!ctx) return;

  // Destroy existing chart if it exists
  if (userChart) {
    userChart.destroy();
  }

  const data = mockData.generateChartData(12);

  userChart = new Chart(ctx, {
    type: 'bar',
    data: {
      labels: data.labels,
      datasets: [{
        label: 'New Users',
        data: data.data.map(d => Math.floor(d / 100)),
        backgroundColor: '#10b981',
        borderRadius: 8
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          display: false
        }
      },
      scales: {
        y: {
          beginAtZero: true
        }
      }
    }
  });
}

// Sales Analytics Chart
function initSalesChart() {
  const ctx = document.getElementById('salesChart');
  if (!ctx) return;

  // Destroy existing chart if it exists
  if (salesChart) {
    salesChart.destroy();
  }

  const data = mockData.generateChartData(12);

  salesChart = new Chart(ctx, {
    type: 'line',
    data: {
      labels: data.labels,
      datasets: [
        {
          label: 'Sales',
          data: data.data,
          borderColor: '#6366f1',
          backgroundColor: 'rgba(99, 102, 241, 0.1)',
          tension: 0.4,
          fill: true
        },
        {
          label: 'Orders',
          data: data.data.map(d => Math.floor(d * 0.8)),
          borderColor: '#8b5cf6',
          backgroundColor: 'rgba(139, 92, 246, 0.1)',
          tension: 0.4,
          fill: true
        }
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        mode: 'index',
        intersect: false
      },
      plugins: {
        legend: {
          position: 'top'
        }
      },
      scales: {
        y: {
          beginAtZero: true
        }
      }
    }
  });
}

// Category Distribution Chart
function initCategoryChart() {
  const ctx = document.getElementById('categoryChart');
  if (!ctx) return;

  // Destroy existing chart if it exists
  if (categoryChart) {
    categoryChart.destroy();
  }

  categoryChart = new Chart(ctx, {
    type: 'doughnut',
    data: {
      labels: ['Electronics', 'Clothing', 'Food', 'Books', 'Others'],
      datasets: [{
        data: [30, 25, 20, 15, 10],
        backgroundColor: [
          '#6366f1',
          '#8b5cf6',
          '#10b981',
          '#f59e0b',
          '#ef4444'
        ]
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          position: 'bottom'
        }
      }
    }
  });
}

// Traffic Sources Chart
function initTrafficChart() {
  const ctx = document.getElementById('trafficChart');
  if (!ctx) return;

  // Destroy existing chart if it exists
  if (trafficChart) {
    trafficChart.destroy();
  }

  trafficChart = new Chart(ctx, {
    type: 'pie',
    data: {
      labels: ['Direct', 'Social Media', 'Search', 'Referral', 'Email'],
      datasets: [{
        data: [35, 25, 20, 12, 8],
        backgroundColor: [
          '#6366f1',
          '#8b5cf6',
          '#10b981',
          '#f59e0b',
          '#3b82f6'
        ]
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          position: 'bottom'
        }
      }
    }
  });
}

// Update chart data
function updateChartData(chart, newData) {
  if (!chart) return;

  chart.data.datasets[0].data = newData;
  chart.update();
}