export class OrderManager {
    constructor(dependencies) {
        this.productManager = dependencies.productManager;
        this.notificationSystem = dependencies.notificationSystem;
        this.customerStorage = dependencies.customerStorage;
        this.settingsManager = dependencies.settingsManager;
        
        this.orderIdCounter = parseInt(localStorage.getItem('lastOrderId') || '0');
    }

    async createOrder(orderData) {
        try {
            // ตรวจสอบสินค้าคงเหลือ
            const stockCheck = await this.checkStock(orderData.items);
            if (!stockCheck.success) {
                return {
                    success: false,
                    error: 'INSUFFICIENT_STOCK',
                    message: stockCheck.message
                };
            }

            // คำนวณค่าจัดส่ง
            const deliveryFee = this.calculateDeliveryFee(orderData);

            // สร้างออเดอร์
            const order = {
                id: this.generateOrderId(),
                ...orderData,
                deliveryFee,
                status: 'pending',
                createdAt: new Date().toISOString()
            };

            // บันทึกออเดอร์
            await this.saveOrder(order);

            // อัพเดทสต็อก
            await this.updateStock(order.items);

            // แจ้งเตือน
            await this.notificationSystem.notifyNewOrder(order);

            // บันทึกประวัติลูกค้า
            this.customerStorage.addToOrderHistory(order);

            return {
                success: true,
                order
            };

        } catch (error) {
            console.error('Order creation failed:', error);
            return {
                success: false,
                error: 'ORDER_CREATION_FAILED',
                message: 'ไม่สามารถสร้างออเดอร์ได้ กรุณาลองใหม่อีกครั้ง'
            };
        }
    }

    generateOrderId() {
        this.orderIdCounter++;
        localStorage.setItem('lastOrderId', this.orderIdCounter.toString());
        return `ORD${String(this.orderIdCounter).padStart(6, '0')}`;
    }

    async checkStock(items) {
        for (const item of items) {
            const product = this.productManager.getProduct(item.id);
            if (!product) {
                return {
                    success: false,
                    message: `สินค้า ${item.name} ไม่พบในระบบ`
                };
            }
            if (product.stock < item.quantity) {
                return {
                    success: false,
                    message: `สินค้า ${item.name} มีไม่เพียงพอ`
                };
            }
        }
        return { success: true };
    }

    calculateDeliveryFee(orderData) {
        const settings = this.settingsManager.getSetting('business.delivery');
        const subtotal = orderData.items.reduce(
            (sum, item) => sum + (item.price * item.quantity), 
            0
        );

        if (subtotal >= settings.freeOver) {
            return 0;
        }

        // คำนวณตามระยะทาง (ตัวอย่าง)
        const distance = this.calculateDistance(orderData.address);
        if (distance > settings.radius) {
            return settings.fee * Math.ceil(distance / settings.radius);
        }

        return settings.fee;
    }

    calculateDistance(address) {
        // ในที่นี้ควรใช้ Google Maps API หรือบริการอื่นๆ
        // สำหรับตัวอย่างจะ return ค่าสมมติ
        return 3; // กิโลเมตร
    }

    async saveOrder(order) {
        const orders = JSON.parse(localStorage.getItem('orders') || '[]');
        orders.push(order);
        localStorage.setItem('orders', JSON.stringify(orders));
    }

    async updateStock(items) {
        items.forEach(item => {
            const product = this.productManager.getProduct(item.id);
            if (product) {
                this.productManager.updateProduct(item.id, {
                    stock: product.stock - item.quantity
                });
            }
        });
    }

    getOrderById(orderId) {
        const orders = JSON.parse(localStorage.getItem('orders') || '[]');
        return orders.find(order => order.id === orderId);
    }

    updateOrderStatus(orderId, status) {
        const orders = JSON.parse(localStorage.getItem('orders') || '[]');
        const orderIndex = orders.findIndex(order => order.id === orderId);
        
        if (orderIndex !== -1) {
            orders[orderIndex].status = status;
            orders[orderIndex].updatedAt = new Date().toISOString();
            localStorage.setItem('orders', JSON.stringify(orders));
            
            // แจ้งเตือนการอัพเดทสถานะ
            this.notificationSystem.notifyStatusUpdate(orders[orderIndex]);
            
            return true;
        }
        return false;
    }
}
