const PaymentManager = {
    init() {
        this.initializeOmise();
        this.setupEventListeners();
    },

    async initializeOmise() {
        // ในแอพจริง โหลด Omise.js
        // const script = document.createElement('script');
        // script.src = 'https://cdn.omise.co/omise.js';
        // document.head.appendChild(script);
    },

    setupEventListeners() {
        // จัดการ event สำหรับฟอร์มชำระเงิน
        document.addEventListener('submit', async (e) => {
            if (e.target.matches('#paymentForm')) {
                e.preventDefault();
                await this.handlePaymentSubmit(e.target);
            }
        });

        // จัดการ event การปิด modal
        document.addEventListener('click', (e) => {
            if (e.target.matches('.modal-close')) {
                const modal = e.target.closest('.modal');
                if (modal) {
                    modal.remove();
                    // ยกเลิกการนับถอยหลัง
                    if (this.paymentTimer) {
                        clearInterval(this.paymentTimer);
                    }
                }
            }
        });
    },

    async handlePaymentSubmit(form) {
        try {
            Utils.showLoading();
            const formData = new FormData(form);
            const paymentMethod = formData.get('paymentMethod');
            
            // สร้างข้อมูล order
            const order = {
                items: CartManager.getCartItems(),
                total: CartManager.getCartTotal().total,
                paymentMethod,
                customer: {
                    name: formData.get('name'),
                    email: formData.get('email'),
                    phone: formData.get('phone'),
                    address: formData.get('address')
                }
            };

            // ดำเนินการชำระเงิน
            const paymentResult = await this.processPayment(order);
            
            if (paymentResult.success) {
                // แสดง modal การชำระเงิน
                this.showPaymentModal(paymentResult);
                // เคลียร์ตะกร้า
                CartManager.clearCart();
                // ปิด modal ชำระเงิน
                document.querySelector('#checkoutModal')?.remove();
            }

        } catch (error) {
            console.error('Payment error:', error);
            NotificationManager.showError(error.message || 'เกิดข้อผิดพลาดในการชำระเงิน');
        } finally {
            Utils.hideLoading();
        }
    },

    async processPayment(order) {
        try {
            switch (order.paymentMethod) {
                case 'promptpay':
                    return await this.processPromptPay(order);
                case 'credit_card':
                    return await this.processCreditCard(order);
                case 'banking':
                    return await this.processBankTransfer(order);
                default:
                    throw new Error('วิธีการชำระเงินไม่ถูกต้อง');
            }
        } catch (error) {
            console.error('Payment processing error:', error);
            throw error;
        }
    },

    async processPromptPay(order) {
        try {
            // สร้าง QR Code
            const qrCode = await this.generatePromptPayQR(order.total);
            const reference = Utils.generateId();
            
            return {
                success: true,
                method: 'promptpay',
                data: {
                    qrCode,
                    amount: order.total,
                    reference,
                    orderDetails: order
                }
            };
        } catch (error) {
            throw new Error('ไม่สามารถสร้าง PromptPay QR Code ได้');
        }
    },

    async processCreditCard(order) {
        // TODO: Implement credit card processing
        throw new Error('ยังไม่รองรับการชำระเงินด้วยบัตรเครดิต');
    },

    async processBankTransfer(order) {
        // TODO: Implement bank transfer processing
        throw new Error('ยังไม่รองรับการชำระเงินด้วยการโอนเงิน');
    },

    showPaymentModal(paymentResult) {
        const modal = document.createElement('div');
        modal.className = 'modal payment-modal active';
        
        const { method, data } = paymentResult;
        const content = this.createPaymentContent(method, data);
        
        modal.innerHTML = `
            <div class="modal-content">
                <button class="modal-close">&times;</button>
                <div class="payment-header">
                    <h2><i class="fas fa-credit-card"></i> ชำระเงิน</h2>
                </div>
                
                ${content}
                
                <div class="payment-footer">
                    <button class="btn btn-secondary modal-close">ปิด</button>
                </div>
            </div>
        `;

        document.body.appendChild(modal);

        // เริ่มนับเวลาถอยหลัง
        if (method === 'promptpay') {
            this.startPaymentTimer(15 * 60); // 15 นาที
        }
    },

    createPaymentContent(method, data) {
        switch (method) {
            case 'promptpay':
                return `
                    <div class="payment-content">
                        <div class="qr-container">
                            <img src="${data.qrCode}" alt="PromptPay QR Code" class="qr-code">
                        </div>
                        
                        <div class="payment-details">
                            <div class="payment-amount">
                                <h3>ยอดชำระ</h3>
                                <span class="amount">${Utils.formatPrice(data.amount)}</span>
                            </div>
                            
                            <div class="payment-reference">
                                <p>เลขอ้างอิง: <strong>${data.reference}</strong></p>
                                <button class="btn btn-small" onclick="Utils.copyToClipboard('${data.reference}')">
                                    <i class="fas fa-copy"></i> คัดลอก
                                </button>
                            </div>
                            
                            <div class="payment-timer">
                                <p>กรุณาชำระเงินภายใน</p>
                                <div class="timer" id="paymentTimer">15:00</div>
                            </div>
                            
                            <div class="payment-instructions">
                                <h4>วิธีการชำระเงิน</h4>
                                <ol>
                                    <li>เปิดแอพธนาคารที่รองรับ PromptPay</li>
                                    <li>เลือกสแกน QR Code</li>
                                    <li>ตรวจสอบยอดเงินและกดยืนยันการชำระเงิน</li>
                                    <li>เก็บหลักฐานการโอนเงินไว้</li>
                                </ol>
                            </div>
                        </div>
                    </div>
                `;
            
            // TODO: เพิ่ม case อื่นๆ สำหรับวิธีการชำระเงินอื่น
            
            default:
                return `<div class="payment-error">ไม่พบวิธีการชำระเงินที่เลือก</div>`;
        }
    },

    startPaymentTimer(duration) {
        let timer = duration;
        const display = document.getElementById('paymentTimer');
        
        // ยกเลิก timer เก่า (ถ้ามี)
        if (this.paymentTimer) {
            clearInterval(this.paymentTimer);
        }
        
        this.paymentTimer = setInterval(() => {
            const minutes = parseInt(timer / 60, 10);
            const seconds = parseInt(timer % 60, 10);

            if (display) {
                display.textContent = 
                    `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                
                // เพิ่ม class เมื่อเวลาน้อย
                if (timer <= 60) { // 1 นาทีสุดท้าย
                    display.classList.add('timer-warning');
                }
            }

            if (--timer < 0) {
                clearInterval(this.paymentTimer);
                this.handlePaymentTimeout();
            }
        }, 1000);
    },

    handlePaymentTimeout() {
        NotificationManager.showError('หมดเวลาชำระเงิน กรุณาทำรายการใหม่');
        document.querySelector('.payment-modal')?.remove();
    }
};
